package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performUpdateAMRRequest(router *Server, amrID int, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("PUT", fmt.Sprintf("http://localhost/v1/amrs/%d", amrID), bytes.NewBuffer(body))
	So(err, ShouldBeNil)

	router.ServeHTTP(w, r)
	return w
}

func TestUpdateAMRs(t *testing.T) {
	Convey("Update AMRs", t, func() {
		Convey("should fail with a bogus body", func() {
			router := mockServer()
			bogusBody := "thecowran"
			w := performUpdateAMRRequest(router, 123, bogusBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid body")
		})

		Convey("should fail with non-existent update fields", func() {
			router := mockServer()
			invalidBody := map[string]interface{}{
				"lost_field": "lost_field_value",
			}
			w := performUpdateAMRRequest(router, 123, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Parameter not allowed: lost_field")
		})

		Convey("should fail with an invalid value for an update field", func() {
			router := mockServer()
			invalidBody := map[string]interface{}{
				"mute_offset_seconds": "should_be_an_integer",
			}
			w := performUpdateAMRRequest(router, 123, invalidBody)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Unrecognized type for integer field: string")
		})

		Convey("should fail for backend error", func() {
			router := mockServer()
			mockBackend.On("UpdateAMR", mock.Anything, 123, mock.Anything).Return(nil, fmt.Errorf("what"))
			validBody := map[string]interface{}{
				"mute_offset_seconds": 12345,
				"song":                "hello",
				"genre":               "rock",
			}
			w := performUpdateAMRRequest(router, 123, validBody)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should succeed with the correct values in the body", func() {
			router := mockServer()
			mockBackend.On("UpdateAMR", mock.Anything, 123, mock.MatchedBy(func(a map[string]interface{}) bool {
				return a["mute_offset_seconds"] == int64(12345) && a["song"] == "hello" && a["genre"] == "rock"
			})).Return(nil, nil)
			validBody := map[string]interface{}{
				"mute_offset_seconds": 12345,
				"song":                "hello",
				"genre":               "rock",
			}
			w := performUpdateAMRRequest(router, 123, validBody)
			So(w.Code, ShouldEqual, http.StatusOK)
		})
	})
}
