package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performUpdateVodRequest(router *Server, vodID interface{}, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest("PUT", fmt.Sprintf("http://localhost/v1/vods/%v", vodID), bytes.NewBuffer(body))
	So(err, ShouldBeNil)

	router.ServeHTTP(w, r)
	return w
}

func TestUpdateVods(t *testing.T) {
	Convey("Update Vods", t, func() {
		Convey("should fail with a bogus body", func() {
			router := mockServer()
			body := "thecowran"
			w := performUpdateVodRequest(router, 123, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid body")
		})

		Convey("should fail when the userID sent is not an integer", func() {
			router := mockServer()
			body := map[string]interface{}{}
			w := performUpdateVodRequest(router, "cow", body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "vod_id is not a number: cow")
		})

		Convey("should fail with invalid keys", func() {
			router := mockServer()
			body := map[string]interface{}{
				"lost_field": "lost_field_value",
			}
			w := performUpdateVodRequest(router, 123, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Parameter not allowed: lost_field")
		})

		Convey("should fail with an invalid value for an update field", func() {
			router := mockServer()
			body := map[string]interface{}{
				"duration": "should_be_an_integer",
			}
			w := performUpdateVodRequest(router, 123, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid body: Unrecognized type for integer field: string")
		})

		Convey("should fail for backend error", func() {
			router := mockServer()
			body := map[string]interface{}{
				"tag_list": "",
			}
			mockBackend.On("UpdateVod", mock.Anything, mock.Anything, int64(123), mock.Anything).Return(nil, fmt.Errorf("what"))

			w := performUpdateVodRequest(router, 123, body)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should succeed with the correct values in the body", func() {
			router := mockServer()
			body := map[string]interface{}{
				"tag_list":    "",
				"title":       "my title",
				"description": "my description",
				"game":        "my game",
				"thumbnails":  "my complicated thumbnails",
			}
			mockBackend.On("UpdateVod", mock.Anything, mock.Anything, int64(123), mock.MatchedBy(func(a models.VodUpdateInput) bool {
				return a.Game.String == body["game"] && a.Description.String == body["description"] && a.Title.String == body["title"]
			})).Return(nil, nil)

			w := performUpdateVodRequest(router, 123, body)
			So(w.Code, ShouldEqual, http.StatusOK)
		})

		Convey("show_formats", func() {
			Convey("fails when not map", func() {
				router := mockServer()
				body := map[string]interface{}{
					"show_formats": "my complicated thumbnails",
				}
				w := performUpdateVodRequest(router, 123, body)
				So(w.Code, ShouldEqual, http.StatusBadRequest)
			})
			Convey("succeeds when empty map", func() {
				router := mockServer()
				mockBackend.On("UpdateVod", mock.Anything, mock.Anything, int64(123), mock.Anything).Return(nil, nil)
				body := map[string]interface{}{
					"show_formats": map[string]interface{}{},
				}
				w := performUpdateVodRequest(router, 123, body)
				So(w.Code, ShouldEqual, http.StatusOK)
			})
		})
	})
}
