package api

import (
	"strconv"
	"strings"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
)

// setBroadcastType checks that broadcastType is valid and sets its default value if unset
func setBroadcastType(broadcastType string) ([]string, error) {
	if broadcastType == "all" {
		return []string{"archive", "highlight"}, nil
	}
	bs := strings.Split(broadcastType, ",")
	for _, b := range bs {
		if b != "archive" && b != "highlight" && b != "upload" {
			return []string{}, errors.InvalidValueError{ParamName: "broadcast_type", Value: b}
		}
	}
	return bs, nil
}

func commaDelimited(param string) []string {
	if param == "" {
		return []string{}
	}
	return strings.Split(param, ",")
}

// validateDurationAndOffset checks that offset is not negative and duration
// is not 0 and returns an error if it is
func validateDurationAndOffset(highlight *models.HighlightInput) error {
	// check offset is not negative
	if highlight.StartSeconds < 0 {
		return errors.InvalidValueError{ParamName: "start_time", Value: highlight.StartSeconds}
	}

	// check duration is not 0
	if (highlight.EndSeconds - highlight.StartSeconds) <= 0 {
		return errors.InvalidDurationError{StartSeconds: highlight.StartSeconds, EndSeconds: highlight.EndSeconds}
	}
	return nil
}

// validateBool tries to convert str to a bool if str is != ""
// returns an error if the conversion fails
func validateBool(str string) (bool, error) {
	var err error
	var res bool
	if str == "" {
		return false, nil
	}

	res, err = strconv.ParseBool(str)
	if err != nil {
		return false, err
	}

	return res, nil
}

// validateNullBool tries to convert str to a bool if str is != ""
// returns an error if the conversion fails
func validateNullBool(str string) (models.NullBool, error) {
	var err error
	var res models.NullBool
	if str == "" {
		return res, nil
	}

	res.Bool, err = strconv.ParseBool(str)
	if err != nil {
		return res, err
	}

	res.Valid = true
	return res, nil
}

// validateNullInt64 tries to convert str to an int64 if str is != ""
// returns an error if the conversion failsA
func validateNullInt64(str string) (models.NullInt64, error) {
	var err error
	var res models.NullInt64
	if str == "" {
		return res, nil
	}

	res.Int64, err = strconv.ParseInt(str, 10, 64)
	if err != nil {
		return res, err
	}

	res.Valid = true
	return res, nil
}
