package api

import (
	"testing"

	. "github.com/smartystreets/goconvey/convey"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
)

func TestSetBroadcastType(t *testing.T) {
	Convey("setBroadcastType", t, func() {
		Convey("should succeed with valid input", func() {
			b, err := setBroadcastType("highlight")
			So(b, ShouldResemble, []string{"highlight"})
			So(err, ShouldBeNil)
		})

		Convey("should succeed with multiple broadcast types", func() {
			b, err := setBroadcastType("upload,highlight")
			So(b, ShouldResemble, []string{"upload", "highlight"})
			So(err, ShouldBeNil)
		})

		Convey("should convert all", func() {
			b, err := setBroadcastType("all")
			So(b, ShouldResemble, []string{"archive", "highlight"})
			So(err, ShouldBeNil)
		})

		Convey("should fail with invalid", func() {

			Convey("empty", func() {
				b, err := setBroadcastType("")
				So(b, ShouldResemble, []string{})
				So(err, ShouldResemble, errors.InvalidValueError{ParamName: "broadcast_type", Value: ""})
			})

			Convey("comma", func() {
				b, err := setBroadcastType(",")
				So(b, ShouldResemble, []string{})
				So(err, ShouldResemble, errors.InvalidValueError{ParamName: "broadcast_type", Value: ""})
			})
			Convey("commas everywhere", func() {
				b, err := setBroadcastType("highlight,,upload")
				So(b, ShouldResemble, []string{})
				So(err, ShouldResemble, errors.InvalidValueError{ParamName: "broadcast_type", Value: ""})
			})
			Convey("highlight,all", func() {
				b, err := setBroadcastType("highlight,all")
				So(b, ShouldResemble, []string{})
				So(err, ShouldResemble, errors.InvalidValueError{ParamName: "broadcast_type", Value: "all"})
			})
			Convey("invalid broadcast type", func() {
				b, err := setBroadcastType("pikachu")
				So(b, ShouldResemble, []string{})
				So(err, ShouldResemble, errors.InvalidValueError{ParamName: "broadcast_type", Value: "pikachu"})
			})
		})
	})
}

func TestCommaDelimited(t *testing.T) {
	Convey("commaDelimited", t, func() {
		Convey("1 value", func() {
			ps := commaDelimited("a")
			So(ps, ShouldResemble, []string{"a"})
		})
		Convey("2 values", func() {
			ps := commaDelimited("a,b")
			So(ps, ShouldResemble, []string{"a", "b"})
		})
		Convey("empty becomes an empty array", func() {
			ps := commaDelimited("")
			So(ps, ShouldResemble, []string{})
		})
		Convey("space", func() {
			ps := commaDelimited(" ")
			So(ps, ShouldResemble, []string{" "})
		})
	})
}
func TestValidateDurationAndOffset(t *testing.T) {
	Convey("validateDurationAndOffset", t, func() {
		Convey("should succeed with valid input", func() {
			err := validateDurationAndOffset(&models.HighlightInput{StartSeconds: int64(0), EndSeconds: int64(100)})
			So(err, ShouldBeNil)
		})

		Convey("should fail with invalid", func() {
			Convey("offset", func() {
				err := validateDurationAndOffset(&models.HighlightInput{StartSeconds: int64(-1), EndSeconds: int64(100)})
				So(err.Error(), ShouldContainSubstring, "Invalid value")
			})

			Convey("duration", func() {
				err := validateDurationAndOffset(&models.HighlightInput{StartSeconds: int64(1000), EndSeconds: int64(1000)})
				So(err.Error(), ShouldContainSubstring, "non-positive duration")
			})
		})
	})
}

func TestValidateBool(t *testing.T) {
	Convey("validateBool", t, func() {
		Convey("should succeed with", func() {
			Convey("empty input", func() {
				res, err := validateBool("")

				So(res, ShouldEqual, false)
				So(err, ShouldBeNil)
			})

			Convey("valid input", func() {
				res, err := validateBool("t")

				So(res, ShouldEqual, true)
				So(err, ShouldBeNil)
			})
		})

		Convey("should fail with", func() {
			Convey("invalid input", func() {
				_, err := validateBool("invalid")

				So(err, ShouldNotBeNil)
				So(err.Error(), ShouldContainSubstring, "ParseBool")
			})
		})
	})
}

func TestValidateNullBool(t *testing.T) {
	Convey("validateNullBool", t, func() {
		Convey("should succeed with", func() {
			Convey("empty input", func() {
				res, err := validateNullBool("")

				So(res.Bool, ShouldEqual, false)
				So(res.Valid, ShouldEqual, false)
				So(err, ShouldBeNil)
			})

			Convey("valid input", func() {
				res, err := validateNullBool("t")

				So(res.Bool, ShouldEqual, true)
				So(res.Valid, ShouldEqual, true)
				So(err, ShouldBeNil)
			})
		})

		Convey("should fail with", func() {
			Convey("invalid input", func() {
				res, err := validateNullBool("invalid")

				So(res.Valid, ShouldEqual, false)
				So(err, ShouldNotBeNil)
				So(err.Error(), ShouldContainSubstring, "ParseBool")
			})
		})
	})
}

func TestValidateNullInt64(t *testing.T) {
	Convey("validateNullInt64", t, func() {
		Convey("should succeed with", func() {
			Convey("empty input", func() {
				res, err := validateNullInt64("")

				So(res.Int64, ShouldEqual, 0)
				So(res.Valid, ShouldEqual, false)
				So(err, ShouldBeNil)
			})

			Convey("valid input", func() {
				res, err := validateNullInt64("123456789")

				So(res.Int64, ShouldEqual, int64(123456789))
				So(res.Valid, ShouldEqual, true)
				So(err, ShouldBeNil)
			})
		})

		Convey("should fail with", func() {
			Convey("invalid input", func() {
				res, err := validateNullInt64("invalid")

				So(res.Valid, ShouldEqual, false)
				So(err, ShouldNotBeNil)
				So(err.Error(), ShouldContainSubstring, "ParseInt")
			})
		})
	})
}
