package api

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func performYoutubeExportRequest(router *Server, vodID interface{}, requestBody interface{}) *httptest.ResponseRecorder {
	w := httptest.NewRecorder()
	body, err := json.Marshal(requestBody)
	So(err, ShouldBeNil)
	r, err := http.NewRequest(http.MethodPost, fmt.Sprintf("http://localhost/v1/vods/%v/youtube_export", vodID), bytes.NewBuffer(body))
	So(err, ShouldBeNil)

	router.ServeHTTP(w, r)
	return w
}

func TestYoutubeExport(t *testing.T) {
	Convey("Youtube export", t, func() {
		Convey("should fail with a bogus body", func() {
			router := mockServer()
			body := "thecowran"
			w := performYoutubeExportRequest(router, 123, body)
			So(w.Code, ShouldEqual, http.StatusBadRequest)
			So(w.Body.String(), ShouldContainSubstring, "Invalid body")
		})

		Convey("should fail for backend error", func() {
			router := mockServer()
			body := map[string]interface{}{
				"tag_list": "",
			}
			mockBackend.On("YoutubeExport", mock.Anything, mock.Anything).Return(fmt.Errorf("what"))

			w := performYoutubeExportRequest(router, 123, body)
			So(w.Code, ShouldEqual, http.StatusInternalServerError)
		})

		Convey("should succeed with the correct values in the body", func() {
			router := mockServer()
			body := map[string]interface{}{
				"title":       "hi",
				"description": "hi",
				"tag_list":    "hi",
				"private":     true,
				"do_split":    true,
			}
			mockBackend.On("YoutubeExport", mock.Anything, mock.Anything).Return(nil)

			w := performYoutubeExportRequest(router, 123, body)
			So(w.Code, ShouldEqual, http.StatusNoContent)
		})

	})
}
