package auth

import (
	"errors"
	"net/http"

	"code.justin.tv/common/goauthorization"
)

// Handler is the interface that auth consumers should be using
type Handler interface {
	AuthorizeJWT(r *http.Request, capabilities goauthorization.CapabilityClaims) error
}

// JwtHandler is default struct implementation with a decoder
type JwtHandler struct {
	Decoder *goauthorization.Decoder
}

// NewHandler returns a new auth handler
func NewHandler(keyPath string) JwtHandler {
	decoder, err := goauthorization.NewDecoder("ES256", keyPath, "code.justin.tv/vod/vinyl", "code.justin.tv/web/cartman")
	if err != nil {
		panic(err)
	}
	return JwtHandler{Decoder: decoder}
}

// AuthorizeJWT checks for the Twitch-Authorization header in the HTTP request
// and validates that the token supplied was signed by the correct issuer.
// It also checks that the attached capability is valid.
func (a JwtHandler) AuthorizeJWT(r *http.Request, capabilities goauthorization.CapabilityClaims) error {
	token, err := a.Decoder.ParseToken(r)
	if err != nil {
		return err
	}
	if err := a.Decoder.Validate(token, capabilities); err != nil {
		return err
	}
	return nil
}

//-------------------------
// Fake Handler for testing
//-------------------------

// FakeHandler is a test struct which is used for tests in the api package.
type FakeHandler struct {
	alwaysApprove bool // configure the fake to always approve auth requests or not (always fail)
}

// NewFakeHandler returns a new fake auth handler, configured to alwaysApprove (true) auth requests or always fail (false)
func NewFakeHandler(alwaysApprove bool) FakeHandler {
	return FakeHandler{alwaysApprove: alwaysApprove}
}

// AuthorizeJWT fake method, that returns no error
func (a FakeHandler) AuthorizeJWT(r *http.Request, capabilities goauthorization.CapabilityClaims) error {
	if a.alwaysApprove {
		return nil
	}
	return errors.New("fake auth failed")
}
