package backend

import (
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"golang.org/x/net/context"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/chat/timing"
	"code.justin.tv/common/golibs/errorlogger"
	connections "code.justin.tv/identity/connections/client"
	"code.justin.tv/vod/vinyl/clients"
	"code.justin.tv/vod/vinyl/datastore"
	dbmodels "code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	"code.justin.tv/vod/vinyl/models"
	"code.justin.tv/vod/vinyl/warnings"
	appSettings "code.justin.tv/web/app-settings/client"
)

const statsdSampleRate = 0.1

// Backender contains functionality that all Backend objects must support.
type Backender interface {
	AcceptUnmoderatedVods(ctx context.Context) error
	CreateAMRs(ctx context.Context, vodID int64, amrs []*models.AMR) ([]*models.AMR, error)
	CreateAppeals(ctx context.Context, vodAppeal *models.VodAppeal, trackAppeals []*models.TrackAppeal) (map[string]interface{}, error)
	CreateHighlight(ctx context.Context, r *http.Request, highlightInfo *models.HighlightInput) (*models.Vod, error)
	CreatePastBroadcast(ctx context.Context, r *http.Request, vod *models.Vod) (*models.Vod, error)
	CreateThumbnails(ctx context.Context, vodID int64, params models.Thumbnails) (models.Thumbnails, error)
	CreateVod(ctx context.Context, r *http.Request, vod *models.Vod) (*models.Vod, error)
	GetUserVideoPrivacyProperties(ctx context.Context, userID int64) (*models.UserVideoPrivacyProperties, error)
	GetUserVideoPrivacyPropertiesBatch(ctx context.Context, userIDs []int64) (map[int64]models.UserVideoPrivacyProperties, error)
	GetUserVODProperties(ctx context.Context, userID int64) (*models.UserVODProperties, error)
	DeleteThumbnail(ctx context.Context, vodID int64, path string) error
	DeleteThumbnails(ctx context.Context, vodID int64) error
	DeleteUserVods(ctx context.Context, userID int64) error
	DeleteVods(ctx context.Context, vodIDs []int64) error
	DeleteVodsInterval(ctx context.Context, startTime time.Time, endTime time.Time) (int, error)
	Followed(ctx context.Context, followedIDs []int, broacastType, language, status []string, sort string, limit, offset int) ([]*models.Vod, error)
	GetVodsByID(ctx context.Context, vodIds []int64, appealsAndAMRs bool, notificationSettings bool, filters *models.VodFilters) ([]*models.Vod, error)
	OwnerForRechat(ctx context.Context, vodID int64) (*models.VodOwner, error)
	GetVodsByUser(ctx context.Context, userID int, broadcastType, language, status []string, appealsAndAMRs bool, notificationSettings bool, filters *models.VodFilters, sort string, limit, offset int) ([]*models.Vod, error)
	GetVodsCountByUser(ctx context.Context, userID int, broadcastType, language, status []string, filters *models.VodFilters) (int64, error)
	GetVodsAggregationByID(ctx context.Context, vodIds []int64, filters *models.VodFilters) (*models.VodsAggregation, error)
	GetAMRsForVod(ctx context.Context, vodID int) ([]*models.AMR, error)
	GetVodAppeals(ctx context.Context, priority, resolved models.NullBool, userInfo string, vodID models.NullInt64, limit, offset int) ([]*models.VodAppeal, error)
	GetVodAppealsCount(ctx context.Context, priority, resolved models.NullBool, userInfo string, vodID models.NullInt64) (int64, error)
	GetVodsByStatus(ctx context.Context, status string, broadcastType []string, startTime, endTime time.Time) ([]*models.Vod, error)
	Moderation(ctx context.Context, vodID int64, statusCode int) error
	PublishScheduledVods(ctx context.Context) error
	Rebroadcast(ctx context.Context, vodIDs []int64, ownerID int64, streamKey string) error
	Related(ctx context.Context, userID, vodID int) ([]*models.Vod, error)
	ResolveTrackAppeal(ctx context.Context, appealID int64, approve bool) error
	ResolveVodAppeal(ctx context.Context, appealID int64) (warnings.Warning, error)
	SetUserVideoPrivacyProperties(ctx context.Context, userID int64, u *models.UserVideoPrivacyPropertiesInput) (*models.UserVideoPrivacyProperties, error)
	SetUserVODProperties(ctx context.Context, userID int64, u *models.UserVODPropertiesInput) (*models.UserVODProperties, error)
	SetViewcounts(ctx context.Context, counts map[int]int64) error
	Top(ctx context.Context, broadcastType, language []string, game, period, sort string, limit, offset int) ([]*models.Vod, error)
	UpdateAMR(ctx context.Context, amrID int, updateFields map[string]interface{}) (*models.AMR, error)
	UpdateVod(ctx context.Context, r *http.Request, vodID int64, u models.VodUpdateInput) (*models.Vod, error)
	YoutubeExport(ctx context.Context, exportInput *models.YoutubeExportInput) error
}

// Backend contains the primary writers and readers to various databases.
type Backend struct {
	Reader             datastore.NamedReader
	MasterReader       datastore.NamedReader
	Writer             datastore.NamedWriter
	Stats              statsd.Statter
	ErrorLogger        errorlogger.ErrorLogger
	PartnershipsClient *clients.PartnershipsHost
	AppSettingsClient  appSettings.Client
	ConnectionsClient  connections.Client
	RailsClient        *clients.RailsHost
	SearchIndexer      *clients.SearchIndexer
	UsersClient        *clients.UsersServiceClient
	Usher              *clients.Usher
	AudreyClient       clients.Audrey
	PushyClient        clients.Pushy
	SpadeClient        clients.SpadeClient
	SearchIndexerLog   *clients.CloudwatchLogStream
}

func convertVods(dbVods []*dbmodels.Vod, options ...dbmodels.AsVinylVodOpt) (models.VodList, error) {
	res := []*models.Vod{}
	for _, dbVod := range dbVods {
		vod, err := dbVod.AsVinylVod(options...)
		if err != nil {
			return nil, err
		}
		res = append(res, vod)
	}
	return res, nil
}

func convertVodOwner(rechatOwner *dbmodels.VodOwner, dbVods []*dbmodels.Vod) (*models.VodOwner, error) {
	res := &models.VodOwner{}
	for _, dbVod := range dbVods {
		vod, err := dbVod.AsVinylVod()
		if err != nil {
			return nil, err
		}
		res.OwnerID = strconv.FormatInt(vod.OwnerID, 10)
		res.StartedOn = vod.StartedOn
		res.Duration = vod.Duration
		res.Status = vod.Status
	}

	res.OwnerLogin = &rechatOwner.OwnerLogin
	return res, nil
}

func convertVodAppeals(dbVodAppeals []*dbmodels.VodAppeal) (models.VodAppealList, error) {
	res := make([]*models.VodAppeal, len(dbVodAppeals))
	for i, dbVodAppeal := range dbVodAppeals {
		vodAppeal, err := dbVodAppeal.AsVinylVodAppeal()
		if err != nil {
			return nil, err
		}
		res[i] = vodAppeal
	}
	return res, nil
}

func convertTrackAppeals(dbTrackAppeals []*dbmodels.TrackAppeal) (models.TrackAppealList, error) {
	res := make([]*models.TrackAppeal, len(dbTrackAppeals))
	for i, dbTrackAppeal := range dbTrackAppeals {
		trackAppeal, err := dbTrackAppeal.AsVinylTrackAppeal()
		if err != nil {
			return nil, err
		}
		res[i] = trackAppeal
	}
	return res, nil
}

func convertAMRs(amrs []*dbmodels.AMR) (models.AMRList, error) {
	res := make([]*models.AMR, len(amrs))
	for i, dbAMR := range amrs {
		amr, err := dbAMR.AsVinylAMR()
		if err != nil {
			return nil, err
		}
		res[i] = amr
	}
	return res, nil
}

func convertVodNotificationSettings(notificationSettingsList []*dbmodels.VodNotificationSettings) (models.VodNotificationSettingsList, error) {
	res := make(models.VodNotificationSettingsList, len(notificationSettingsList))
	for i, dbNotificationSettings := range notificationSettingsList {
		notificationSettings, err := dbNotificationSettings.AsVinylVodNotificationSettings()
		if err != nil {
			return nil, err
		}
		res[i] = notificationSettings
	}
	return res, nil
}

func (b *Backend) time(backendName string, action string, f func(context.Context) error) {
	xact := &timing.Xact{Stats: b.Stats}
	statName := fmt.Sprintf("secondary.%s.%s", statsdNormalize(backendName), statsdNormalize(action))
	xact.AddName(statName)
	xact.Start()
	ctx := timing.XactContext(context.Background(), xact)

	err := f(ctx)
	if err == nil {
		xact.End("success")
	} else {
		xact.End("failure")
	}
}

func statsdNormalize(s string) string {
	return strings.Replace(strings.ToLower(s), ".", "-", -1)
}

func includeUnderReview(status []string) []string {
	for _, s := range status {
		if s == models.StatusTranscoding {
			return append(status, models.StatusUnderReview)
		}
	}
	return status
}
