package backend

import (
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
)

// CreateAppeals calls the appropriate writers to store track and vod appeal data.
func (b *Backend) CreateAppeals(ctx context.Context, vodAppeal *models.VodAppeal, trackAppeals []*models.TrackAppeal) (map[string]interface{}, error) {
	priority, err := b.getPriority(ctx, vodAppeal.VodID)
	if err != nil {
		return nil, err
	}

	vodAppeal.CreatedAt = time.Now().UTC().Round(time.Second)
	vodAppeal.UpdatedAt = time.Now().UTC().Round(time.Second)
	vodAppeal.Priority = priority

	vodAppeal.VodAppealID, err = b.Writer.CreateVodAppeal(ctx, vodAppeal, false)
	if err != nil {
		return map[string]interface{}{}, err
	}

	for _, ta := range trackAppeals {
		ta.VodAppealID = vodAppeal.VodAppealID
		ta.CreatedAt = time.Now().UTC().Round(time.Second)
		ta.UpdatedAt = time.Now().UTC().Round(time.Second)
	}

	ids, err := b.Writer.CreateTrackAppeals(ctx, trackAppeals, vodAppeal.VodAppealID, false)
	if err != nil {
		return map[string]interface{}{}, err
	}

	for i, id := range ids {
		trackAppeals[i].TrackAppealID = id
		go b.trackMuteAppealRequestSuccess(trackAppeals[i], vodAppeal.VodID)
	}

	return map[string]interface{}{
		"vod_appeal":    vodAppeal,
		"track_appeals": trackAppeals,
	}, nil
}

// getPriority gets the owner of vodID and queries the partnership service to check
// whether or not the owner is in the partner program and returns true / false accordingly
func (b *Backend) getPriority(ctx context.Context, vodID int64) (bool, error) {
	vods, err := b.GetVodsByID(ctx, []int64{vodID}, false, false, models.VodFilterNone())
	if err != nil {
		return false, err
	}
	if len(vods) == 0 {
		return false, errors.NotFoundError{Type: "Vod", ID: int(vodID)}
	}

	properties, err := b.PartnershipsClient.GetPartnerProperties(ctx, vods[0].OwnerID)
	if err != nil {
		return false, err
	}

	if properties.PartnerProgram.Valid {
		return properties.PartnerProgram.Bool, nil
	}
	return false, nil
}

func (b *Backend) trackMuteAppealRequestSuccess(t *models.TrackAppeal, vodID int64) {
	event := map[string]interface{}{
		"amr_id": t.AudibleMagicResponseID,
		"vod_id": vodID,
	}

	b.SpadeClient.SendEvent("vod_mute_appeal_request", event)
}
