package backend

import (
	"strings"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"

	"golang.org/x/net/context"
)

// CreateThumbnails creates thumbnails for the given vod
func (b *Backend) CreateThumbnails(ctx context.Context, vodID int64, params models.Thumbnails) (models.Thumbnails, error) {
	if params == nil {
		return nil, errors.MissingParamError{ParamName: "thumbnails"}
	}

	// Transcoder is sending thumbnails with a null entry in some requests,
	// For now we will clean up the input and fulfill the request, but we
	// should fix transcoder and reject requests with invalid input in the future
	params = removeNullThumbnails(params)

	for _, thumbnailParams := range params {
		if err := thumbnailParams.Validate(); err != nil {
			return nil, err
		}
	}

	thumbnails, err := b.Writer.CreateThumbnails(ctx, vodID, params)
	if err != nil {
		return nil, err
	}

	err = b.UpdateVodSearchIndexer(ctx, vodID)
	if err != nil {
		return nil, err
	}

	go b.trackThumbnailCreate(vodID, thumbnails)

	return thumbnails, nil
}

func removeNullThumbnails(thumbnails models.Thumbnails) models.Thumbnails {
	clean := models.Thumbnails{}
	for _, thumb := range thumbnails {
		if thumb != nil {
			clean = append(clean, thumb)
		}
	}
	return clean
}

func (b *Backend) trackThumbnailCreate(vodID int64, thumbnails models.Thumbnails) {
	var paths []string
	for _, thumbnail := range thumbnails {
		paths = append(paths, thumbnail.Path)
	}

	attributes := map[string]interface{}{}
	attributes["vod_id"] = vodID
	attributes["paths"] = strings.Join(paths, ",")
	b.SpadeClient.SendEvent("vod_thumbnail_create", attributes)
}
