package backend

import (
	"fmt"
	"net/http"
	"time"
	"unicode/utf8"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

// MaxAllowableConcurrentUploads is the max number of concurrent uploads a user is allowed to have at one time.
var MaxAllowableConcurrentUploads = int64(15)

// CreateVod calls the appropriate writers to store vod data.
func (b *Backend) CreateVod(ctx context.Context, r *http.Request, vod *models.Vod) (*models.Vod, error) {
	n, err := b.Reader.GetTranscodingVodsCountByUser(ctx, vod.OwnerID)
	if err != nil {
		// If the rate limit query fails, continue normally
		fmt.Println("Unable to check # of currently transcoding vods:", err)
	} else if n >= MaxAllowableConcurrentUploads {
		return nil, errors.TooManyUploadsError{}
	}

	vod.CreatedAt = time.Now().UTC().Round(time.Second)
	vod.UpdatedAt = time.Now().UTC().Round(time.Second)
	vod.StartedOn = time.Now().UTC().Round(time.Second)
	if vod.BroadcastType == "" {
		vod.BroadcastType = "archive"
	}
	vod.Views = 0
	vod.Offset = 0
	vod.CreatedBy = models.NullInt64{Valid: true, Int64: int64(vod.OwnerID)}
	vod.Origin = models.NullString{String: "s3", Valid: true}
	if !vod.Title.Valid || vod.Title.String == "" {
		return nil, errors.TitleTooShortError{}
	}
	if vod.Title.Valid && utf8.RuneCountInString(vod.Title.String) > 100 {
		return nil, errors.TitleTooLongError{}
	}

	createdVod, err := b.Writer.CreatePastBroadcast(ctx, vod, false)
	if err != nil {
		return nil, err
	}

	res, err := createdVod.AsVinylVod()
	if err != nil {
		return nil, err
	}
	go b.trackVodInit(res, r)

	return res, nil
}
