package backend

import (
	"golang.org/x/net/context"

	"code.justin.tv/vod/vinyl/models"
)

func (b *Backend) includeAppealsAndAMRs(ctx context.Context, res models.VodList) error {
	if len(res) == 0 {
		return nil
	}

	vodIDs := make([]int64, len(res))
	for index, vod := range res {
		vodIDs[index] = vod.ID
	}

	vodAppeals, trackAppeals, amrs, err := b.getAppealsAndAMRs(ctx, vodIDs)
	if err != nil {
		return err
	}

	// idVodsMap stores (vodID, vod) to easily find a vod by id and insert the
	// vod appeals and amrs that correspond to it
	// vodAppealIDVodsMap stores (vodAppealID, vod) so a track appeal can lookup which
	// vod corresponds to its vod appeal id
	idVodsMap, vodAppealIDVodsMap := map[int64]*models.Vod{}, map[int64]*models.Vod{}
	// whether each vod currently has a pending appeal
	hasPendingAppealMap := map[int64]bool{}

	// store all vods into the map so a vod can be found by its id
	for _, vod := range res {
		idVodsMap[vod.ID] = vod
	}

	// for each amr find the vod in idVodsMap that matches amr.VodID and append
	// the amr to vod.AudibleMagicResponses
	for _, amr := range amrs {
		vodID := amr.VodID
		vod := idVodsMap[vodID]
		vod.AudibleMagicResponses = append(vod.AudibleMagicResponses, amr)
	}

	// for each vod appeal, find the vod in idVodsMap that matches va.VodID and
	// append the vod appeal to vod.VodAppeals. store the vod into vodAppealIDVodsMap
	// so track appeals can look up a vod by its vod appeal id
	for _, va := range vodAppeals {
		vodID := va.VodID
		vod := idVodsMap[vodID]
		vod.VodAppeal = va
		hasPendingAppealMap[vodID] = va.ResolvedAt.Invalid()

		vaID := va.VodAppealID
		vodAppealIDVodsMap[vaID] = vod
	}

	// for each track appeal, find the vod in vodAppealIDVodsMap that matches
	// ta.VodAppealID and append the track appeal to vod.TrackAppeals
	for _, ta := range trackAppeals {
		vaID := ta.VodAppealID
		vod := vodAppealIDVodsMap[vaID]
		vod.TrackAppeals = append(vod.TrackAppeals, ta)
	}

	// set some flags based on appeal status
	// TODO: move these to web-client
	for _, vod := range res {
		isMuted := len(vod.AudibleMagicResponses) > 0
		vod.IsMuted = models.NullBool{Bool: isMuted, Valid: true}
		if isMuted {
			hasPendingAppeal := hasPendingAppealMap[vod.ID]
			vod.CanCreateAppeal = models.NullBool{Bool: !hasPendingAppeal, Valid: true}
			vod.ViewOnlyAppeal = models.NullBool{Bool: hasPendingAppeal, Valid: true}
			vod.HasPendingAppeal = models.NullBool{Bool: hasPendingAppeal, Valid: true}
		}
	}
	return nil
}

// getAppealsAndAMRs calls the appropriate backend functions to fetch the track
// appeals, vod appeals, and audible magic responses given vodIDs
func (b *Backend) getAppealsAndAMRs(ctx context.Context, vodIDs []int64) ([]*models.VodAppeal, []*models.TrackAppeal, []*models.AMR, error) {
	var vodAppeals []*models.VodAppeal
	var trackAppeals []*models.TrackAppeal
	var amrs []*models.AMR
	var err error

	dbVodAppeals, err := b.Reader.GetVodAppealsByVodIDs(ctx, vodIDs)
	if err != nil {
		return vodAppeals, trackAppeals, amrs, err
	}

	vodAppeals, err = convertVodAppeals(dbVodAppeals)
	if err != nil {
		return vodAppeals, trackAppeals, amrs, err
	}

	if len(vodAppeals) > 0 {
		vodAppealIDs := make([]int64, len(vodAppeals))
		for i, vodAppeal := range vodAppeals {
			vodAppealIDs[i] = vodAppeal.VodAppealID
		}

		dbTrackAppeals, err := b.Reader.GetTrackAppealsByVodAppealIDs(ctx, vodAppealIDs)
		if err != nil {
			return vodAppeals, trackAppeals, amrs, err
		}

		trackAppeals, err = convertTrackAppeals(dbTrackAppeals)
		if err != nil {
			return vodAppeals, trackAppeals, amrs, err
		}
	}

	dbAMRs, err := b.Reader.GetAMRsByVodIDs(ctx, vodIDs)
	if err != nil {
		return vodAppeals, trackAppeals, amrs, err
	}

	amrs, err = convertAMRs(dbAMRs)
	if err != nil {
		return vodAppeals, trackAppeals, amrs, err
	}
	return vodAppeals, trackAppeals, amrs, nil
}
