package backend

import (
	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

const (
	acceptStatusCode      = 1
	rejectStatusCode      = 2
	legalActionStatusCode = 3
	brokenStatusCode      = 4
)

var (
	rejectedStatus    = "rejected"
	legalActionStatus = "legal_action"

	acceptModeration      = "moderation.accept"
	rejectModeration      = "moderation.reject"
	legalActionModeration = "moderation.legal_action"
	brokenModeration      = "moderation.broken"
	labels                = []string{acceptModeration, rejectModeration, legalActionModeration, brokenModeration}
)

//Moderation updates the VOD as published, soft deleted, or hard deleted according to statusCode
func (b *Backend) Moderation(ctx context.Context, vodID int64, statusCode int) error {
	statsdLabel := labels[statusCode-1]
	statsErr := b.Stats.Inc(statsdLabel, 1, 1.0)
	if statsErr != nil {
	}

	switch statusCode {
	case acceptStatusCode, brokenStatusCode:
		u := models.VodUpdateInput{
			Status: models.NullString{String: models.StatusRecorded, Valid: true},
		}
		_, err := b.UpdateVod(ctx, nil, vodID, u)
		if err != nil {
			return err
		}
	case rejectStatusCode:
		vod, err := b.UpdateVod(ctx, nil, vodID, models.VodUpdateInput{
			Deleted: models.NullBool{Bool: true, Valid: true},
		})
		if err != nil {
			return err
		}

		err = b.PushyClient.SendModerationEmail(ctx, vod.OwnerID, vodID, rejectedStatus)
		if err != nil {
			return err
		}
	case legalActionStatusCode:
		vods, err := b.GetVodsByID(ctx, []int64{vodID}, false, false, models.VodFilterNone())
		if err != nil {
			return err
		}

		if len(vods) == 0 {
			return errors.NotFoundError{Type: "Vod", ID: int(vodID)}
		}

		err = b.PushyClient.SendModerationEmail(ctx, vods[0].OwnerID, vodID, legalActionStatus)
		if err != nil {
			return err
		}
	}
	return nil
}
