package backend

import (
	edgemodels "code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

const numUpdatesInParallel = 10

type updateVodResult struct {
	vod *edgemodels.Vod
	err error
}

// PublishScheduledVods makes vods scheduled through viewable at
func (b *Backend) PublishScheduledVods(ctx context.Context) error {
	dbVods, err := b.Reader.GetScheduledVods(ctx)
	if err != nil {
		return err
	}
	if len(dbVods) == 0 {
		return nil
	}

	update := edgemodels.VodUpdateInput{
		Viewable:   edgemodels.NullString{String: "public", Valid: true},
		ViewableAt: edgemodels.NullTime{Time: nil, Present: true},
	}

	results := make(chan *updateVodResult, len(dbVods))
	c := make(chan struct{}, numUpdatesInParallel)
	for _, dbVod := range dbVods {
		c <- struct{}{} // blocks if buffer full until goroutine does <-c
		go func() {
			defer func() {
				_ = <-c
			}()

			vod, err := dbVod.AsVinylVod()
			if err != nil {
				results <- &updateVodResult{
					err: err,
				}
				return
			}

			vod, err = b.UpdateVod(ctx, nil, vod.ID, update)
			results <- &updateVodResult{
				vod: vod,
				err: err,
			}
		}()
	}

	vods := make([]*edgemodels.Vod, 0, len(dbVods))
	for i := 0; i < len(dbVods); i++ {
		result := <-results
		if result.err != nil {
			err = result.err
		} else {
			vods = append(vods, result.vod)
		}
	}

	if len(vods) > 0 {
		go b.SearchIndexerAddBatch(context.Background(), vods)
	}

	return err
}
