package backend

import (
	"strings"

	"code.justin.tv/vod/vinyl/clients"
	vinylerrors "code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
	"code.justin.tv/vod/vinyl/warnings"

	"golang.org/x/net/context"
)

// ResolveVodAppeal calls the relevant writers to modify the vod appeal resolve date and sends an email to the user
// informing them of the vod appeal status.
func (b *Backend) ResolveVodAppeal(ctx context.Context, appealID int64) (warnings.Warning, error) {
	vodID, err := b.Writer.ResolveVodAppeal(ctx, appealID)
	if err != nil {
		return nil, err
	}

	// Compose an email to the user informing them the vod appeal has completed
	vods, err := b.GetVodsByID(ctx, []int64{vodID}, true, false, models.VodFilterDeleted())
	if err != nil {
		return nil, err
	}
	if len(vods) == 0 {
		return nil, vinylerrors.NotFoundError{Type: "Vod", ID: int(vodID)}
	}

	vod := vods[0]
	trackAppeals := vod.TrackAppeals
	amrs := vod.AudibleMagicResponses

	trackInfo := []map[string]interface{}{}
	for _, tAppeal := range trackAppeals {
		trackData := map[string]interface{}{
			"title":     "",
			"performer": "",
			"reason":    strings.Title(strings.Join(strings.Split(tAppeal.Reason, "_"), " ")),
		}
		for _, amr := range amrs {
			if amr.AudibleMagicResponseID == tAppeal.AudibleMagicResponseID {
				if amr.Title.Valid {
					trackData["title"] = amr.Title.String
				}
				if amr.Performer.Valid {
					trackData["performer"] = amr.Performer.String
				}
				trackData["muted"] = amr.UnmutedAt.Invalid()
				break
			}
		}
		trackInfo = append(trackInfo, trackData)
	}

	mailBody := map[string]interface{}{}
	mailBody["track_appeals"] = trackInfo

	userInfo, err := b.UsersClient.GetUserInfo(ctx, int64(vod.OwnerID))
	if err != nil {
		return nil, err
	}

	// Validate user info before sending it so we can return better error responses to the users.
	// Ideally, we would not have to do this because the error from sending the email would be robust enough.
	warning := validateUserInfo(userInfo)
	if warning != nil {
		return warning, nil
	}

	mailBody["login"] = userInfo.DisplayName

	err = b.RailsClient.SendEmail(ctx, int64(vod.OwnerID), "audio_appeal_result", mailBody)

	return nil, err
}

func validateUserInfo(userInfo clients.UserInfo) warnings.Warning {
	email := userInfo.Email.String
	if !userInfo.Email.Valid {
		return warnings.InvalidEmailAddressWarning{}
	}

	// requirement for activision, site-wide for consistency.
	if email == "" {
		return warnings.InvalidEmailAddressWarning{}
	}
	if userInfo.DeletedOn.Present && userInfo.DeletedOn.Time != nil {
		return warnings.DeletedUserWarning{}
	}
	if userInfo.TermsOfServiceViolation.Valid && userInfo.TermsOfServiceViolation.Bool {
		return warnings.TOSWarning{}
	}
	if !userInfo.EmailVerified.Valid || !userInfo.EmailVerified.Bool {
		return warnings.UnverifiedEmailWarning{}
	}
	return nil
}
