package backend

import (
	"testing"
	"time"

	"code.justin.tv/vod/vinyl/clients"
	"code.justin.tv/vod/vinyl/models"
	. "github.com/smartystreets/goconvey/convey"
)

func TestValidateUserInfo(t *testing.T) {
	Convey("Testing User info validation", t, func() {
		validUser := clients.UserInfo{
			UserID:                  1,
			Login:                   models.NullString{String: "login", Valid: true},
			DisplayName:             models.NullString{String: "displayname", Valid: true},
			DmcaViolation:           models.NullBool{Bool: false, Valid: true},
			TermsOfServiceViolation: models.NullBool{Bool: false, Valid: true},
			DeletedOn:               models.NullTime{},
			Language:                models.NullString{String: "language", Valid: true},
			Category:                models.NullString{String: "category", Valid: true},
			RemoteIP:                models.NullString{String: "remoteIP", Valid: true},
			Email:                   models.NullString{String: "email", Valid: true},
			LastLogin:               models.NullString{String: "lastLogin", Valid: true},
			BannedUntil:             models.NullTime{},
			DMCAViolationCount:      models.NullInt64{Int64: int64(0), Valid: true},
			TOSViolationCount:       models.NullInt64{Int64: int64(0), Valid: true},
			EmailVerified:           models.NullBool{Bool: true, Valid: true}}

		Convey("Validating a legitimate user should not result in any errors", func() {
			userInfo := validUser
			warning := validateUserInfo(userInfo)
			So(warning, ShouldEqual, nil)
		})

		Convey("Validating a user without an email should result in an error", func() {
			userInfo := validUser
			userInfo.Email = models.NullString{String: "", Valid: true}
			warning := validateUserInfo(userInfo)
			So(warning.Message(), ShouldEqual, "Email was not sent because user does not have a valid email address.")
		})

		Convey("Validating a deleted user should result in an error", func() {
			now := time.Now()
			userInfo := validUser
			userInfo.DeletedOn = models.NullTime{Time: &now, Present: true}
			warning := validateUserInfo(userInfo)
			So(warning.Message(), ShouldEqual, "Email was not sent because user has been deleted.")
		})

		Convey("Validating a user who violated TOS should result in an error", func() {
			userInfo := validUser
			userInfo.TermsOfServiceViolation = models.NullBool{Bool: true, Valid: true}
			warning := validateUserInfo(userInfo)
			So(warning.Message(), ShouldEqual, "Email was not sent because user has violated the terms of service.")
		})
	})
}
