package backend

import (
	"encoding/json"
	"fmt"
	"strconv"
	"strings"

	"golang.org/x/net/context"

	"code.justin.tv/vod/vinyl/clients"
	"code.justin.tv/vod/vinyl/models"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/kinesis"
)

// SearchIndexerAdd adds a vod to searchindexer
func (b *Backend) SearchIndexerAdd(ctx context.Context, vod *models.Vod) {
	b.SearchIndexerAddBatch(ctx, []*models.Vod{vod})
}

// SearchIndexerRemove removes a vod from the searchindex
func (b *Backend) SearchIndexerRemove(ctx context.Context, vod *models.Vod) {
	b.SearchIndexerRemoveBatch(ctx, []int64{vod.ID})
}

// SearchIndexerAddBatch batch adds vods to searchindexer
func (b *Backend) SearchIndexerAddBatch(ctx context.Context, vods []*models.Vod) {
	userMetadata := b.getUserMetadata(ctx, vods)

	watchableVods := []*models.Vod{}
	userIDs := []int64{}
	for _, vod := range vods {
		if vod.Watchable() {
			watchableVods = append(watchableVods, vod)
			userIDs = append(userIDs, vod.OwnerID)
		}
	}

	userPrivacyProperties, err := b.GetUserVideoPrivacyPropertiesBatch(ctx, userIDs)
	if err != nil {
		return
	}
	watchablePublicVods := []*models.Vod{}
	for _, vod := range watchableVods {
		hideArchives := userPrivacyProperties[vod.OwnerID].HideArchives
		if !hideArchives {
			watchablePublicVods = append(watchablePublicVods, vod)
		}
	}
	if len(watchablePublicVods) == 0 {
		return
	}

	var records []*kinesis.PutRecordsRequestEntry
	for _, vod := range watchablePublicVods {
		data, err := json.Marshal(map[string]interface{}{
			"action": "update",
			"document": map[string]interface{}{
				"id":     strconv.FormatInt(vod.ID, 10),
				"fields": fields(vod, userMetadata),
				"type":   "vod",
			},
		})
		if err == nil {
			records = append(records, &kinesis.PutRecordsRequestEntry{
				Data:         data,
				PartitionKey: aws.String(strconv.FormatInt(vod.ID, 10)),
			})
		}
	}
	b.SearchIndexer.Index(records, "add_vods")

	go func() {
		logs := []string{}
		for _, vod := range watchablePublicVods {
			logs = append(logs, strconv.FormatInt(vod.ID, 10)+" added to searchindexer")
		}
		err := b.SearchIndexerLog.Log(logs)
		if err != nil {
			fmt.Println("Unable to log to cloudwatch:", err)
		}
	}()
}

// SearchIndexerRemoveBatch batch removes vods from searchindexer
func (b *Backend) SearchIndexerRemoveBatch(ctx context.Context, vodIDs []int64) {
	var records []*kinesis.PutRecordsRequestEntry
	for _, id := range vodIDs {
		data, err := json.Marshal(map[string]interface{}{
			"action": "delete",
			"document": map[string]interface{}{
				"id":     strconv.FormatInt(id, 10),
				"fields": []interface{}{},
				"type":   "vod",
			},
		})
		if err == nil {
			records = append(records, &kinesis.PutRecordsRequestEntry{
				Data:         data,
				PartitionKey: aws.String(strconv.FormatInt(id, 10)),
			})
		}
	}
	b.SearchIndexer.Index(records, "remove_vods")

	go func() {
		logs := []string{}
		for _, id := range vodIDs {
			logs = append(logs, strconv.FormatInt(id, 10)+" removed from searchindexer")
		}
		err := b.SearchIndexerLog.Log(logs)
		if err != nil {
			fmt.Println("Unable to log to cloudwatch:", err)
		}
	}()
}

func (b *Backend) getUserMetadata(ctx context.Context, vods []*models.Vod) map[int64]*clients.UserInfo {
	userMetadata := map[int64]*clients.UserInfo{}

	ownerIDs := []int64{}
	for _, vod := range vods {
		ownerIDs = append(ownerIDs, int64(vod.OwnerID))
	}

	userInfo, err := b.UsersClient.GetUserInfoBatch(ctx, ownerIDs)
	if err != nil {
		return userMetadata
	}
	for _, u := range userInfo {
		userMetadata[u.UserID] = u
	}
	return userMetadata
}

func fields(vod *models.Vod, userMetadata map[int64]*clients.UserInfo) []map[string]interface{} {
	var (
		login, displayName string
		time               int64
	)
	if userInfo, ok := userMetadata[vod.OwnerID]; ok {
		if userInfo.Login.Valid {
			login = userInfo.Login.String
		} else {
			fmt.Printf("User does not have a login. User info: %v", userInfo)
		}
		if userInfo.DisplayName.Valid {
			displayName = userInfo.DisplayName.String
		} else {
			fmt.Printf("User does not have a display name. User info: %v", userInfo)
		}
	}

	broadcastTypePriority, _ := map[string]int{
		"archive":   100,
		"highlight": 200,
		"upload":    300,
	}[vod.BroadcastType]

	if vod.PublishedAt.Present {
		time = vod.PublishedAt.Time.Unix()
	} else {
		time = vod.StartedOn.Unix()
	}

	interpolatedPreview := strings.Replace(vod.PreviewTemplate, "%{width}", "208", 1)
	interpolatedPreview = strings.Replace(interpolatedPreview, "%{height}", "117", 1)
	return []map[string]interface{}{
		{"name": "broadcast_type", "value": vod.BroadcastType, "type": "enum"},
		{"name": "broadcast_type_priority", "value": broadcastTypePriority, "type": "integer"},
		{"name": "broadcaster_id", "value": strconv.FormatInt(vod.OwnerID, 10), "type": "string"},
		{"name": "broadcaster_login", "value": login, "type": "string"},
		{"name": "broadcaster_name", "value": displayName, "type": "string"},
		{"name": "created_at", "value": time, "type": "integer"},
		{"name": "description", "value": vod.Description.String, "type": "text"},
		{"name": "game", "value": vod.Game.String, "type": "string"},
		{"name": "language", "value": vod.Language.String, "type": "string"},
		{"name": "length", "value": vod.Duration, "type": "integer"},
		{"name": "tags", "value": strings.Split(vod.TagList, ", "), "type": "list"},
		{"name": "thumbnail", "value": interpolatedPreview, "type": "enum"},
		{"name": "title", "value": vod.Title.String, "type": "text"},
		{"name": "views", "value": vod.Views, "type": "integer"},
	}
}
