package backend

import (
	"testing"
	"time"

	"code.justin.tv/vod/vinyl/clients"
	"code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
)

func TestFields(t *testing.T) {
	now := time.Now().Round(time.Second)
	vod := &models.Vod{
		BroadcastType:   "highlight",
		OwnerID:         64,
		PublishedAt:     models.NullTime{Present: true, Time: &now},
		Description:     models.NullString{Valid: true, String: "team kill"},
		Game:            models.NullString{Valid: true, String: "Overwatch"},
		Duration:        10,
		Language:        models.NullString{Valid: true, String: "en"},
		TagList:         "a, b, c, d",
		PreviewTemplate: "file-%{width}x%{height}.jpg",
		Title:           models.NullString{Valid: true, String: "Play of the game"},
		Views:           100000,
	}
	userMetadata := map[int64]*clients.UserInfo{64: &clients.UserInfo{
		Login:       models.NullString{Valid: true, String: "twitch"},
		DisplayName: models.NullString{Valid: true, String: "Twitch"}}}

	expected := []map[string]interface{}{
		{"name": "broadcast_type", "value": "highlight", "type": "enum"},
		{"name": "broadcast_type_priority", "value": 200, "type": "integer"},
		{"name": "broadcaster_id", "value": "64", "type": "string"},
		{"name": "broadcaster_login", "value": "twitch", "type": "string"},
		{"name": "broadcaster_name", "value": "Twitch", "type": "string"},
		{"name": "created_at", "value": now.Unix(), "type": "integer"},
		{"name": "description", "value": "team kill", "type": "text"},
		{"name": "game", "value": "Overwatch", "type": "string"},
		{"name": "language", "value": "en", "type": "string"},
		{"name": "length", "value": 10, "type": "integer"},
		{"name": "tags", "value": []string{"a", "b", "c", "d"}, "type": "list"},
		{"name": "thumbnail", "value": "file-208x117.jpg", "type": "enum"},
		{"name": "title", "value": "Play of the game", "type": "text"},
		{"name": "views", "value": int64(100000), "type": "integer"},
	}

	Convey("Test fields", t, func() {
		res := fields(vod, userMetadata)
		for i := range res {
			So(res[i], ShouldResemble, expected[i])
		}
	})

	Convey("invalid broadcast type", t, func() {
		vod.BroadcastType = "asdf"
		res := fields(vod, userMetadata)
		expected[1]["value"] = 0
		expected[0]["value"] = "asdf"
		for i := range res {
			So(res[i], ShouldResemble, expected[i])
		}
	})
}
