package backend

import (
	"log"
	"net/http"
	"time"
	"unicode/utf8"

	"code.justin.tv/vod/vinyl/errors"
	edgemodels "code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

// UpdateVod calls the appropriate writers to update a vod.
func (b *Backend) UpdateVod(ctx context.Context, r *http.Request, vodID int64, u edgemodels.VodUpdateInput) (*edgemodels.Vod, error) {
	// check if the VOD is watchable prior to performing any changes
	vods, err := b.MasterReader.GetVodsByID(ctx, []int64{vodID}, edgemodels.VodFilterNone())
	if err != nil {
		return nil, err
	}
	if len(vods) == 0 {
		return nil, errors.NotFoundError{Type: "Vod", ID: int(vodID)}
	}

	prev, err := convertVods(vods)
	if err != nil {
		return nil, err
	}

	vodWasWatchable := prev[0].Watchable()
	prevStatus := prev[0].Status

	// Set updated at field.
	u.UpdatedAt = time.Now().UTC().Round(time.Second)

	if u.Title.Valid {
		if u.Title.String == "" {
			return nil, errors.TitleTooShortError{}
		} else if utf8.RuneCountInString(u.Title.String) >= 100 {
			return nil, errors.TitleTooLongError{}
		}
	}
	//Set published_at time if its first time viewable and never published before
	if prev[0].BroadcastType == "upload" && !prev[0].PublishedAt.Present && b.vodWillBeWatchable(prev[0], u) {
		u.PublishedAt = edgemodels.NullTime{Time: &u.UpdatedAt, Present: true}
	}

	if u.SelectedThumbnailPath.Valid {
		err = b.Writer.UpdateSelectedThumbnail(ctx, vodID, u.SelectedThumbnailPath.String)
		if err != nil {
			log.Println(err)
		}
	}

	// A crude state machine for protection against invalid state transitions. For now,
	// the assumption is that 'recorded' is a terminal state.
	if u.Status.Valid && prevStatus == edgemodels.StatusRecorded && prevStatus != u.Status.String {
		return nil, errors.InvalidStateTransitionError{prevStatus, u.Status.String}
	}

	// We don't merely return the IDs of the updated broadcasts due to the thumbnails field.
	// It is passed in as a serialized string, but Vinyl needs to return it as a parsed map.
	vod, err := b.Writer.UpdateVod(ctx, vodID, &u)
	if err != nil {
		return nil, err
	}

	// Update VOD's notification settings
	// load existing notification settings and then overwrite with new settings
	notificationSettingsMap := edgemodels.VodNotificationSettingsMap{}
	notificationSettingsList, err := b.getVodNotificationSettings(ctx, []int64{vodID})
	for _, notificationSettings := range notificationSettingsList {
		err = b.addNotificationSettingsToMap(notificationSettings, &notificationSettingsMap)
		if err != nil {
			return nil, err
		}
	}
	if u.Notifications.Email != nil {
		err = b.updateAndIncludeVodNotificationSettings(ctx, vodID, "email", u.Notifications.Email, &notificationSettingsMap)
		if err != nil {
			return nil, err
		}
	}
	if u.Notifications.ChannelFeed != nil {
		err = b.updateAndIncludeVodNotificationSettings(ctx, vodID, "channel_feed", u.Notifications.ChannelFeed, &notificationSettingsMap)
		if err != nil {
			return nil, err
		}
	}

	res, err := vod.AsVinylVod()
	if err != nil {
		return nil, err
	}

	// we consider sending notifications for a *watchable* VOD in these cases:
	// - the VOD was not watchable before the update
	// - notification settings were changed
	vodIsWatchable := res.Watchable()

	notificationSettingsList = notificationSettingsMap.AsList()
	if vodIsWatchable && (!vodWasWatchable || len(notificationSettingsList) > 0) {
		if err != nil {
			return nil, err
		}

		// send notifications for enabled notifieres
		for _, notificationSettings := range notificationSettingsList {
			if notificationSettings.Enabled {
				now := time.Now().UTC().Round(time.Second)
				disabledNotificationSettings := &edgemodels.VodNotificationSettingsInput{
					Enabled: false,
					SentAt:  edgemodels.NullTime{Time: &now, Present: true},
				}
				updatedNotificationSettings, err := b.Writer.UpdateVodNotificationSettings(ctx, vodID, notificationSettings.Type, disabledNotificationSettings)
				if err != nil {
					return nil, err
				}

				notificationSettings, err = updatedNotificationSettings.AsVinylVodNotificationSettings()
				if err != nil {
					return nil, err
				}

				err = b.sendNotifications(ctx, res, notificationSettings)
				if err != nil {
					return nil, err
				}

				err = b.addNotificationSettingsToMap(notificationSettings, &notificationSettingsMap)
				if err != nil {
					return nil, err
				}
			}
		}
	}

	res.NotificationSettings = notificationSettingsMap

	if vodIsWatchable {
		go b.SearchIndexerAdd(context.Background(), res)
	} else {
		go b.SearchIndexerRemove(context.Background(), res)
	}

	if prevStatus != edgemodels.StatusRecorded && res.Status == edgemodels.StatusRecorded {
		go b.trackVodCreateSuccess(res)
	}
	if prevStatus != edgemodels.StatusFailed && res.Status == edgemodels.StatusFailed {
		go b.trackVodCreateFailure(res)
	}

	if prevStatus == res.Status {
		go b.trackVodEdit(res, r)
	} else {
		go b.trackVodCreationStatus(res)
	}
	return res, nil
}

func (b *Backend) vodWillBeWatchable(vod *edgemodels.Vod, u edgemodels.VodUpdateInput) bool {

	copyVod := *vod
	if u.Viewable.Valid {
		copyVod.Viewable.Valid = u.Viewable.Valid
		copyVod.Viewable.String = u.Viewable.String
	}
	if u.Status.Valid {
		copyVod.Status = u.Status.String
	}
	if u.Deleted.Valid {
		copyVod.Deleted.Valid = u.Deleted.Valid
		copyVod.Deleted.Bool = u.Deleted.Bool
	}

	return copyVod.Watchable()

}

func (b *Backend) updateAndIncludeVodNotificationSettings(ctx context.Context, vodID int64, notificationType string, uNotificationSettings *edgemodels.VodNotificationSettingsInput, notificationSettingsMap *edgemodels.VodNotificationSettingsMap) error {
	datastoreNotificationSettings, err := b.Writer.UpdateVodNotificationSettings(ctx, vodID, notificationType, uNotificationSettings)
	if err != nil {
		return err
	}

	notificationSettings, err := datastoreNotificationSettings.AsVinylVodNotificationSettings()
	if err != nil {
		return err
	}

	err = b.addNotificationSettingsToMap(notificationSettings, notificationSettingsMap)
	if err != nil {
		return err
	}

	return nil
}
