package backend

import (
	"fmt"
	"strconv"

	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

// YoutubeExport exports a vod to youtube
func (b *Backend) YoutubeExport(ctx context.Context, input *models.YoutubeExportInput) error {
	youtubeExportDisabled, err := b.AppSettingsClient.GetBool(ctx, "web", "disable_youtube_exporter", nil)
	if err != nil {
		// if call to AppSettings could not be completed, continue execution assuming the feature is enabled
	} else if youtubeExportDisabled {
		return errors.YoutubeExportFeatureDisabledError{}
	}

	i, err := strconv.ParseInt(input.VodID, 10, 64)
	if err != nil {
		return err
	}

	vods, err := b.Reader.GetVodsByID(ctx, []int64{i}, nil)
	if err != nil {
		return err
	}
	vod := vods[0]

	ytUser, err := b.ConnectionsClient.GetYoutubeUser(ctx, fmt.Sprintf("%v", vod.OwnerID), nil)
	if err != nil {
		return err
	}

	userInfo, err := b.UsersClient.GetUserInfo(ctx, vod.OwnerID)
	if err != nil || !userInfo.Login.Valid {
		return err
	}

	err = b.Usher.YoutubeExport(input, ytUser, userInfo.Login.String)
	if err != nil {
		// The user will get emailed by the worker if it fails to process the job
	}

	b.trackYoutubeExportSuccess(vod.ID)
	return nil
}

func (b *Backend) trackYoutubeExportSuccess(vodID int64) {
	event := map[string]interface{}{
		"vod_id": vodID,
	}

	b.SpadeClient.SendEvent("youtube_export", event)
}
