package vinyl

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
)

const (
	defaultStatSampleRate = 0.1
	defaultTimingXactName = "vinyl"
)

// Client is the interface a Vinyl client implements
type Client interface {
	GetVodByID(ctx context.Context, id string, params *GetVodParams, reqOpts *twitchhttp.ReqOpts) (*Vod, error)
	GetVodsByIDs(ctx context.Context, ids []string, params *GetVodParams, reqOpts *twitchhttp.ReqOpts) ([]*Vod, error)
	GetVodsAggregationByIDs(ctx context.Context, ids []string, params *GetVodParams, reqOpts *twitchhttp.ReqOpts) (*VodsAggregation, error)
	Moderation(ctx context.Context, vodID, status string, reqOpts *twitchhttp.ReqOpts) error
	DeleteVodExternal(ctx context.Context, vodID string, reqOpts *twitchhttp.ReqOpts) error
	DeleteVodsExternal(ctx context.Context, vodIDs []string, reqOpts *twitchhttp.ReqOpts) error
	Top(ctx context.Context, broadcastType, language, game, period, sort string, limit int, offset int, reqOpts *twitchhttp.ReqOpts) ([]*Vod, error)
	Followed(ctx context.Context, ids []string, broadcastType, language, sort string, limit int, offset int, reqOpts *twitchhttp.ReqOpts) ([]*Vod, error)
	GetVodsByUser(ctx context.Context, userID, broadcastType, language, status, sort string, limit, offset int, params *GetVodParams, reqOpts *twitchhttp.ReqOpts) (*GetVodsByUserResponse, error)
	CreateHighlight(ctx context.Context, input CreateHighlightInput, reqOpts *twitchhttp.ReqOpts) (*Vod, error)
	Rebroadcast(ctx context.Context, vodIDs []string, ownerID, streamKey string, reqOpts *twitchhttp.ReqOpts) error
	CreateAppeals(ctx context.Context, input CreateAppealsInput, reqOpts *twitchhttp.ReqOpts) error
	YoutubeExport(ctx context.Context, input YoutubeExportInput, reqOpts *twitchhttp.ReqOpts) error
}

type client struct {
	twitchhttp.Client
}

// NewClient takes a twitchhttp.ClientConf and returns a client implementing the Vinyl client interface
func NewClient(conf twitchhttp.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchhttp.NewClient(conf)
	return &client{twitchClient}, err
}

type getVodsByIDsResponse struct {
	TotalCount int    `json:"total_count"`
	Vods       []*Vod `json:"vods"`
}

// GetVodsByUserResponse is the response of the /v1/vods/user/:userID endpoint
type GetVodsByUserResponse struct {
	TotalCount int    `json:"total_count"`
	Vods       []*Vod `json:"vods"`
}

// VodList is a list of vods
type VodList []*Vod

type createHighlightResponse struct {
	Vod *Vod `json:"highlight"`
}

// GetVodParams will translate into request parameters used when loading /v1/vods
// Please use NewGetVodParams to instantiate with default values.
type GetVodParams struct {
	WithMuteInfo             bool // (appeals_and_amrs) response with mute info fields (i.e. muted_segments)
	WithNotificationSettings bool // (notification_settings) response with notification_settings fields
	IncludeDeleted           bool // (include_deleted)
	IncludePrivate           bool // (include_private)
	IncludeProcessing        bool // (include_processing)
}

// NewGetVodParams instantiates GetVodParams with defaults
func NewGetVodParams() *GetVodParams {
	return &GetVodParams{
		WithMuteInfo:             false, // since needs extra effort to load this data
		WithNotificationSettings: false,
		IncludeDeleted:           false, // soft-removed vods are ignored by default
		IncludePrivate:           true,
		IncludeProcessing:        true,
	}
}

// URLQueryString converts a GetVodParam struct into the expected query parameters for the vinyl backend
func (p *GetVodParams) URLQueryString() string {
	qs := fmt.Sprintf(
		"appeals_and_amrs=%t&notification_settings=%t&include_deleted=%t&include_private=%t&include_processing=%t",
		p.WithMuteInfo,
		p.WithNotificationSettings,
		p.IncludeDeleted,
		p.IncludePrivate,
		p.IncludeProcessing,
	)
	return qs
}

func (c *client) GetVodByID(ctx context.Context, idStr string, params *GetVodParams, reqOpts *twitchhttp.ReqOpts) (*Vod, error) {
	if params == nil {
		params = &GetVodParams{} // defaults
	}

	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods/%s?%s", idStr, params.URLQueryString()), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.get_vod_by_id",
		StatSampleRate: defaultStatSampleRate,
	})

	vod := Vod{}
	resp, err := c.DoJSON(ctx, &vod, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if resp.StatusCode == 404 {
		return nil, &twitchhttp.Error{StatusCode: http.StatusNotFound, Message: "vod not found"}
	}

	return &vod, nil
}

func (c *client) GetVodsByIDs(ctx context.Context, ids []string, params *GetVodParams, reqOpts *twitchhttp.ReqOpts) ([]*Vod, error) {
	if params == nil {
		params = &GetVodParams{} // defaults
	}
	idsStr := strings.Join(ids, ",")

	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods?ids=%s&%s", idsStr, params.URLQueryString()), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.get_vods_by_ids",
		StatSampleRate: defaultStatSampleRate,
	})

	data := getVodsByIDsResponse{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	if len(data.Vods) == 0 {
		return nil, &twitchhttp.Error{StatusCode: http.StatusNotFound, Message: "vods not found"}
	}

	return data.Vods, nil
}

func (c *client) GetVodsAggregationByIDs(ctx context.Context, ids []string, params *GetVodParams, reqOpts *twitchhttp.ReqOpts) (*VodsAggregation, error) {
	if params == nil {
		params = &GetVodParams{} // defaults
	}
	idsStr := strings.Join(ids, ",")

	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods_aggregation?ids=%s&%s", idsStr, params.URLQueryString()), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.get_vods_aggregation_by_ids",
		StatSampleRate: defaultStatSampleRate,
	})

	data := VodsAggregation{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return &data, nil
}

func (c *client) DeleteVodsExternal(ctx context.Context, vodIDs []string, reqOpts *twitchhttp.ReqOpts) error {
	vodIDsStr := strings.Join(vodIDs, ",")
	req, err := c.NewRequest(http.MethodDelete, fmt.Sprintf("/v1/vods/external?ids=%s", vodIDsStr), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.delete_vods_external",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
		}
	}()

	if resp.StatusCode >= 400 {
		return twitchhttp.HandleFailedResponse(resp)
	}

	return nil
}

func (c *client) DeleteVodExternal(ctx context.Context, vodID string, reqOpts *twitchhttp.ReqOpts) error {
	req, err := c.NewRequest(http.MethodDelete, fmt.Sprintf("/v1/vods/external?ids=%s", vodID), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.delete_vod_external",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
		}
	}()

	if resp.StatusCode >= 400 {
		return twitchhttp.HandleFailedResponse(resp)
	}

	return nil
}

func (c *client) Moderation(ctx context.Context, vodID, status string, reqOpts *twitchhttp.ReqOpts) error {
	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods/moderation/%s?status=%s", vodID, status), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.moderate_vod",
		StatSampleRate: defaultStatSampleRate,
	})

	resp, err := c.Do(ctx, req, combinedReqOpts)
	if err != nil {
		return err
	}
	defer func() {
		err := resp.Body.Close()
		if err != nil {
		}
	}()

	if resp.StatusCode >= 400 {
		return twitchhttp.HandleFailedResponse(resp)
	}

	return nil
}

func (c *client) Top(ctx context.Context, broadcastType, language, game, period, sort string, limit int, offset int, reqOpts *twitchhttp.ReqOpts) ([]*Vod, error) {
	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods/top?broadcast_type=%s&language=%s&game=%s&period=%s&sort=%s&limit=%d&offset=%d", broadcastType, language, url.QueryEscape(game), period, sort, limit, offset), nil)
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.top",
		StatSampleRate: defaultStatSampleRate,
	})

	data := VodList{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

func (c *client) Followed(ctx context.Context, ids []string, broadcastType, language, sort string, limit int, offset int, reqOpts *twitchhttp.ReqOpts) ([]*Vod, error) {
	var idsAsInt []int

	for _, stringID := range ids {
		id, err := strconv.Atoi(stringID)
		if err == nil {
			idsAsInt = append(idsAsInt, id)
		}
	}

	i, err := json.Marshal(idsAsInt)
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, fmt.Sprintf("/v1/vods/followed?broadcast_type=%s&language=%s&sort=%s&limit=%d&offset=%d", broadcastType, language, sort, limit, offset), bytes.NewBuffer(i))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.followed",
		StatSampleRate: defaultStatSampleRate,
	})

	data := VodList{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data, err
}

func (c *client) CreateHighlight(ctx context.Context, input CreateHighlightInput, reqOpts *twitchhttp.ReqOpts) (*Vod, error) {
	i, err := json.Marshal(map[string]CreateHighlightInput{"highlight": input})
	if err != nil {
		return nil, err
	}

	req, err := c.NewRequest(http.MethodPost, "/v1/vods/highlight", bytes.NewBuffer(i))
	if err != nil {
		return nil, err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.create_highlight",
		StatSampleRate: defaultStatSampleRate,
	})

	data := createHighlightResponse{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return data.Vod, err
}

// CreateHighlightInput contains the possible input parameters that go into a CreateHighlight operation.
type CreateHighlightInput struct {
	VodID        int64  `json:"id"`
	Description  string `json:"description"`
	Game         string `json:"game"`
	Title        string `json:"title"`
	Language     string `json:"language"`
	StartSeconds int    `json:"start_time"`
	EndSeconds   int    `json:"end_time"`
	TagList      string `json:"tag_list"`
	CreatedBy    int64  `json:"created_by_id"`
}

func (c *client) CreateAppeals(ctx context.Context, input CreateAppealsInput, reqOpts *twitchhttp.ReqOpts) error {
	i, err := json.Marshal(input)
	if err != nil {
		return err
	}

	req, err := c.NewRequest(http.MethodPost, "/v1/appeals", bytes.NewBuffer(i))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.create_appeals",
		StatSampleRate: defaultStatSampleRate,
	})

	data := struct{}{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return err
}

// CreateAppealsInput is the input format associated with the CreateAppeals endpoint.
type CreateAppealsInput struct {
	VodID        string        `json:"vod_id"`
	VodAppeal    VodAppeal     `json:"vod_appeal_params"`
	TrackAppeals []TrackAppeal `json:"track_appeals"`
}

func (c *client) YoutubeExport(ctx context.Context, input YoutubeExportInput, reqOpts *twitchhttp.ReqOpts) error {
	i, err := json.Marshal(input)
	if err != nil {
		return err
	}

	req, err := c.NewRequest(http.MethodPost, "/v1/vods/:vod_id/youtube_export", bytes.NewBuffer(i))
	if err != nil {
		return err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.youtube_export",
		StatSampleRate: defaultStatSampleRate,
	})

	data := struct{}{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return err
}

// YoutubeExportInput is the input format associated with the YoutubeExport endpoint.
type YoutubeExportInput struct {
	VodID       string `json:"vod_id"`
	Title       string `json:"title"`
	Description string `json:"description"`
	TagList     string `json:"tag_list"`
	Private     bool   `json:"private"`
	DoSplit     bool   `json:"do_split"`
}

func (c *client) GetVodsByUser(ctx context.Context, userID, broadcastType, language, status, sort string, limit, offset int, params *GetVodParams, reqOpts *twitchhttp.ReqOpts) (*GetVodsByUserResponse, error) {
	if params == nil {
		params = &GetVodParams{} // defaults
	}

	req, err := c.NewRequest(http.MethodGet, fmt.Sprintf("/v1/vods/user/%s", userID), nil)
	if err != nil {
		return nil, err
	}

	q := req.URL.Query()
	q.Add("broadcast_type", broadcastType)
	q.Add("language", language)
	q.Add("status", status)
	q.Add("sort", sort)
	q.Add("appeals_and_amrs", fmt.Sprintf("%t", params.WithMuteInfo))
	q.Add("notification_settings", fmt.Sprintf("%t", params.WithNotificationSettings))
	q.Add("include_deleted", fmt.Sprintf("%t", params.IncludeDeleted))
	q.Add("include_private", fmt.Sprintf("%t", params.IncludePrivate))
	q.Add("include_processing", fmt.Sprintf("%t", params.IncludeProcessing))
	q.Add("limit", fmt.Sprintf("%d", limit))
	q.Add("offset", fmt.Sprintf("%d", offset))
	req.URL.RawQuery = q.Encode()

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.get_vods_by_user",
		StatSampleRate: defaultStatSampleRate,
	})

	data := GetVodsByUserResponse{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	if err != nil {
		return nil, err
	}

	return &data, nil
}

func (c *client) Rebroadcast(ctx context.Context, ids []string, ownerID, streamKey string, reqOpts *twitchhttp.ReqOpts) error {
	idsStr := strings.Join(ids, ",")
	req, err := c.NewRequest(http.MethodPost, fmt.Sprintf("/v1/vods/rebroadcast?ids=%s&owner_id=%s&stream_key=%s", idsStr, ownerID, streamKey), nil)
	if err != nil {
		return err
	}

	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       "service.vinyl.rebroadcast",
		StatSampleRate: defaultStatSampleRate,
	})

	data := struct{}{}
	_, err = c.DoJSON(ctx, &data, req, combinedReqOpts)
	return err
}
