package vinyl

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"golang.org/x/net/context"

	"code.justin.tv/common/twitchhttp"
	. "github.com/smartystreets/goconvey/convey"
)

const errorResponse = `
{
  "error": "Missing required param: ids",
  "status": 400
}
`

const moderationError = `
{
  "error": "Missing required param: vod_id",
  "status": 400
}
`

const singleVodResponse = `
{
  "broadcast_id": 22999100432,
  "broadcast_type": "archive",
  "broadcaster_software": null,
  "created_at": "2016-08-25T17:01:53Z",
  "created_by_id": 25871845,
  "delete_at": null,
  "deleted": null,
  "description": null,
  "duration": 28517,
  "fps": {
    "audio_only": 0,
    "chunked": 60.00003226138115,
    "high": 30.000208997643096,
    "low": 30.000279131080376,
    "medium": 30.000208997643096,
    "mobile": 30.000279131080376
  },
  "game": "Hearthstone: Heroes of Warcraft",
  "id": 85635764,
  "language": "en",
  "manifest": "index-dvr.m3u8",
  "offset": 0,
  "origin": null,
  "owner_id": 25871845,
  "resolutions": {
    "chunked": "1920x1080",
    "high": "1280x720",
    "low": "640x360",
    "medium": "852x480",
    "mobile": "400x226"
  },
  "source_archive_id": null,
  "recorded_on": "2016-08-25T17:01:37Z",
  "status": "recorded",
  "tag_list": "",
  "title": "Kibler - Karazhan Week Three!",
  "total_length": 28517,
  "updated_at": "2016-08-26T00:57:05Z",
  "uri": "v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573",
  "viewable": "public",
  "viewable_at": null,
  "views": 67193,
  "increment_view_count_url": "https://countess.twitch.tv/ping.gif?u=%7B%22id%22%3A%2285635764%22%2C%22type%22%3A%22vod%22%7D",
  "qualities": [
    "audio_only",
    "chunked",
    "high",
    "low",
    "medium",
    "mobile"
  ],
  "preview_template": "https://static-cdn.jtvnw.net/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/thumb/thumb0-%{width}x%{height}.jpg",
  "show_formats": {
    "audio_only": {
      "bitrate": 178302,
      "codecs": "mp4a.40.2",
      "display_name": "Audio Only",
      "fps": 0,
      "playlist_preference": 5
    },
    "chunked": {
      "bitrate": 3226463,
      "codecs": "avc1.4D402A,mp4a.40.2",
      "display_name": "Source",
      "fps": 60.00003226138115,
      "playlist_preference": 0,
      "resolution": "1920x1080"
    },
    "high": {
      "bitrate": 1399536,
      "codecs": "avc1.4D401F,mp4a.40.2",
      "display_name": "High",
      "fps": 30.000208997643096,
      "playlist_preference": 1,
      "resolution": "1280x720"
    },
    "low": {
      "bitrate": 588823,
      "codecs": "avc1.42C01E,mp4a.40.2",
      "display_name": "Low",
      "fps": 30.000279131080376,
      "playlist_preference": 3,
      "resolution": "640x360"
    },
    "medium": {
      "bitrate": 795774,
      "codecs": "avc1.4D401E,mp4a.40.2",
      "display_name": "Medium",
      "fps": 30.000208997643096,
      "playlist_preference": 2,
      "resolution": "852x480"
    },
    "mobile": {
      "bitrate": 280478,
      "codecs": "avc1.42C00D,mp4a.40.2",
      "display_name": "Mobile",
      "fps": 30.000279131080376,
      "playlist_preference": 4,
      "resolution": "400x226"
    }
  },
  "thumbnail_templates": [
    {
      "offset": 26717.349,
      "path": "thumb/thumb0.jpg",
      "type": "generated",
      "url": "https://static-cdn.jtvnw.net/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/thumb/thumb0-%{width}x%{height}.jpg"
    },
    {
      "offset": 16517.615999999998,
      "path": "thumb/thumb1.jpg",
      "type": "generated",
      "url": "https://static-cdn.jtvnw.net/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/thumb/thumb1-%{width}x%{height}.jpg"
    },
    {
      "offset": 28213.349,
      "path": "thumb/thumb2.jpg",
      "type": "generated",
      "url": "https://static-cdn.jtvnw.net/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/thumb/thumb2-%{width}x%{height}.jpg"
    },
    {
      "offset": 27317.349,
      "path": "thumb/thumb3.jpg",
      "type": "generated",
      "url": "https://static-cdn.jtvnw.net/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/thumb/thumb3-%{width}x%{height}.jpg"
    }
  ],
  "seek_previews_url": "https://vod-storyboards.twitch.tv/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/storyboards/85635764-info.json",
  "animated_preview_url": "https://vod-storyboards.twitch.tv/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/storyboards/85635764-strip-0.jpg",
  "path": "/videos/85635764",
  "url": "https://twitch.tv/videos/85635764",
  "vod_appeal": null,
  "track_appeals": null,
  "is_muted": null,
  "can_create_appeal": null,
  "view_only_appeal": null,
  "has_pending_appeal": null,
  "api_id": "v85635764"
}
`

const multiVodReponse = `

{
  "vods": [
    {
      "broadcast_id": 16611790528,
      "broadcast_type": "highlight",
      "broadcaster_software": null,
      "created_at": "2015-09-25T16:32:02Z",
      "created_by_id": 12826,
      "delete_at": null,
      "deleted": null,
      "description": "Join Marcus \"djWHEAT\" Graham and Twitch CEO Emmett Shear for the TwitchCon keynote speech, covering the past, present, and future of Twitch.",
      "duration": 2748,
      "fps": {
        "audio_only": 0,
        "chunked": 60,
        "high": 30.0625,
        "low": 30.0625,
        "medium": 30.0625,
        "mobile": 20
      },
      "game": "TwitchCon 2015",
      "id": 17714623,
      "language": "en",
      "manifest": "highlight-17714623-muted-L0J9G2H7OY.m3u8",
      "offset": 1639,
      "origin": "s3",
      "owner_id": 12826,
      "resolutions": {
        "chunked": "1280x720",
        "high": "1280x720",
        "low": "640x360",
        "medium": "852x480",
        "mobile": "400x226"
      },
      "source_archive_id": 17691616,
      "recorded_on": "2015-09-25T16:58:56Z",
      "status": "recorded",
      "tag_list": "",
      "title": "TwitchCon 2015 - Twitch Keynote",
      "total_length": 2748,
      "updated_at": "2016-05-22T18:04:22Z",
      "uri": "8d2c492f9d_twitch_16611790528_313721965",
      "viewable": "public",
      "viewable_at": null,
      "views": 3693,
      "increment_view_count_url": "https://countess.twitch.tv/ping.gif?u=%7B%22id%22%3A%2217714623%22%2C%22type%22%3A%22vod%22%7D",
      "qualities": [
        "audio_only",
        "chunked",
        "high",
        "low",
        "medium",
        "mobile"
      ],
      "preview_template": "https://static-cdn.jtvnw.net/s3_vods/8d2c492f9d_twitch_16611790528_313721965//thumb/thumb17714623-%{width}x%{height}.jpg",
      "show_formats": {
        "audio_only": {
          "bitrate": 213098,
          "codecs": "mp4a.40.2",
          "display_name": "Audio Only",
          "fps": 0.0,
          "playlist_preference": 5
        },
        "chunked": {
          "bitrate": 3766486,
          "codecs": "avc1.4D4029,mp4a.40.2",
          "display_name": "Source",
          "fps": 60.0,
          "playlist_preference": 0,
          "resolution": "1280x720"
        },
        "high": {
          "bitrate": 1580516,
          "codecs": "avc1.42C01F,mp4a.40.2",
          "display_name": "High",
          "fps": 30.0625,
          "playlist_preference": 1,
          "resolution": "1280x720"
        },
        "low": {
          "bitrate": 657530,
          "codecs": "avc1.42C01E,mp4a.40.2",
          "display_name": "Low",
          "fps": 30.0625,
          "playlist_preference": 3,
          "resolution": "640x360"
        },
        "medium": {
          "bitrate": 901930,
          "codecs": "avc1.42C01E,mp4a.40.2",
          "display_name": "Medium",
          "fps": 30.0625,
          "playlist_preference": 2,
          "resolution": "852x480"
        },
        "mobile": {
          "bitrate": 300800,
          "codecs": "avc1.42C00D,mp4a.40.2",
          "display_name": "Mobile",
          "fps": 20.0,
          "playlist_preference": 4,
          "resolution": "400x226"
        }
      },
      "thumbnail_templates": [
        {
          "offset": 0,
          "path": "/thumb/thumb17714623.jpg",
          "type": "generated",
          "url": "https://static-cdn.jtvnw.net/s3_vods/8d2c492f9d_twitch_16611790528_313721965//thumb/thumb17714623-%{width}x%{height}.jpg"
        }
      ],
      "seek_previews_url": "https://vod-storyboards.twitch.tv/8d2c492f9d_twitch_16611790528_313721965/storyboards/17714623-info.json",
      "animated_preview_url": "https://vod-storyboards.twitch.tv/8d2c492f9d_twitch_16611790528_313721965/storyboards/17714623-strip-0.jpg",
      "path": "/videos/17714623",
      "url": "https://twitch.tv/videos/17714623",
      "vod_appeal": null,
      "track_appeals": null,
      "is_muted": null,
      "can_create_appeal": null,
      "view_only_appeal": null,
      "has_pending_appeal": null,
      "api_id": "v17714623"
    },
    {
      "broadcast_id": 22999100432,
      "broadcast_type": "archive",
      "broadcaster_software": null,
      "created_at": "2016-08-25T17:01:53Z",
      "created_by_id": 25871845,
      "delete_at": null,
      "deleted": null,
      "description": null,
      "duration": 28517,
      "fps": {
        "audio_only": 0,
        "chunked": 60.00003226138115,
        "high": 30.000208997643096,
        "low": 30.000279131080376,
        "medium": 30.000208997643096,
        "mobile": 30.000279131080376
      },
      "game": "Hearthstone: Heroes of Warcraft",
      "id": 85635764,
      "language": "en",
      "manifest": "index-dvr.m3u8",
      "offset": 0,
      "origin": null,
      "owner_id": 25871845,
      "resolutions": {
        "chunked": "1920x1080",
        "high": "1280x720",
        "low": "640x360",
        "medium": "852x480",
        "mobile": "400x226"
      },
      "source_archive_id": null,
      "recorded_on": "2016-08-25T17:01:37Z",
      "status": "recorded",
      "tag_list": "",
      "title": "Kibler - Karazhan Week Three!",
      "total_length": 28517,
      "updated_at": "2016-08-26T00:57:05Z",
      "uri": "v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573",
      "viewable": "public",
      "viewable_at": null,
      "views": 66781,
      "increment_view_count_url": "https://countess.twitch.tv/ping.gif?u=%7B%22id%22%3A%2285635764%22%2C%22type%22%3A%22vod%22%7D",
      "qualities": [
        "audio_only",
        "chunked",
        "high",
        "low",
        "medium",
        "mobile"
      ],
      "preview_template": "https://static-cdn.jtvnw.net/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/thumb/thumb0-%{width}x%{height}.jpg",
      "show_formats": {
        "audio_only": {
          "bitrate": 178302,
          "codecs": "mp4a.40.2",
          "display_name": "Audio Only",
          "fps": 0,
          "playlist_preference": 5
        },
        "chunked": {
          "bitrate": 3226463,
          "codecs": "avc1.4D402A,mp4a.40.2",
          "display_name": "Source",
          "fps": 60.00003226138115,
          "playlist_preference": 0,
          "resolution": "1920x1080"
        },
        "high": {
          "bitrate": 1399536,
          "codecs": "avc1.4D401F,mp4a.40.2",
          "display_name": "High",
          "fps": 30.000208997643096,
          "playlist_preference": 1,
          "resolution": "1280x720"
        },
        "low": {
          "bitrate": 588823,
          "codecs": "avc1.42C01E,mp4a.40.2",
          "display_name": "Low",
          "fps": 30.000279131080376,
          "playlist_preference": 3,
          "resolution": "640x360"
        },
        "medium": {
          "bitrate": 795774,
          "codecs": "avc1.4D401E,mp4a.40.2",
          "display_name": "Medium",
          "fps": 30.000208997643096,
          "playlist_preference": 2,
          "resolution": "852x480"
        },
        "mobile": {
          "bitrate": 280478,
          "codecs": "avc1.42C00D,mp4a.40.2",
          "display_name": "Mobile",
          "fps": 30.000279131080376,
          "playlist_preference": 4,
          "resolution": "400x226"
        }
      },
      "thumbnail_templates": [
        {
          "offset": 26717.349,
          "path": "thumb/thumb0.jpg",
          "type": "generated",
          "url": "https://static-cdn.jtvnw.net/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/thumb/thumb0-%{width}x%{height}.jpg"
        },
        {
          "offset": 16517.615999999998,
          "path": "thumb/thumb1.jpg",
          "type": "generated",
          "url": "https://static-cdn.jtvnw.net/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/thumb/thumb1-%{width}x%{height}.jpg"
        },
        {
          "offset": 28213.349,
          "path": "thumb/thumb2.jpg",
          "type": "generated",
          "url": "https://static-cdn.jtvnw.net/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/thumb/thumb2-%{width}x%{height}.jpg"
        },
        {
          "offset": 27317.349,
          "path": "thumb/thumb3.jpg",
          "type": "generated",
          "url": "https://static-cdn.jtvnw.net/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/thumb/thumb3-%{width}x%{height}.jpg"
        }
      ],
      "seek_previews_url": "https://vod-storyboards.twitch.tv/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/storyboards/85635764-info.json",
      "animated_preview_url": "https://vod-storyboards.twitch.tv/v1/AUTH_system/vods_a978/bmkibler_22999100432_506167573/storyboards/85635764-strip-0.jpg",
      "path": "/videos/85635764",
      "url": "https://twitch.tv/videos/85635764",
      "vod_appeal": null,
      "track_appeals": null,
      "is_muted": null,
      "can_create_appeal": null,
      "view_only_appeal": null,
      "has_pending_appeal": null,
      "api_id": "v85635764"
    }
  ]
}
`

const vodsAggregationResponse = `
{
	"total_count": 2,
	"total_duration": 45
}
`

func TestGetVodByID(t *testing.T) {
	Convey("When calling GetVodByID", t, func() {
		Convey("when the service responds with OK status", func() {
			// mock a valid response
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, singleVodResponse)
			}))
			client, _ := NewClient(twitchhttp.ClientConf{Host: ts.URL})

			vod, err := client.GetVodByID(context.Background(), "322", nil, nil)
			Convey("returns a VOD object", func() {
				So(err, ShouldBeNil)
				So(*vod.Title, ShouldEqual, "Kibler - Karazhan Week Three!")
				So(vod.Path, ShouldEqual, "/videos/85635764")
				So(vod.URL, ShouldEqual, "https://twitch.tv/videos/85635764")
			})
		})
		Convey("when the service responds with an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, errorResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			_, err = client.GetVodByID(context.Background(), "322", nil, nil)
			Convey("returns an error", func() {
				So(err, ShouldNotBeNil)
			})
		})
		Convey("when using GetVodParams and the service responds with OK status", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, singleVodResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			params := NewGetVodParams()
			params.WithMuteInfo = true
			params.IncludeDeleted = true
			vod, err := client.GetVodByID(context.Background(), "322", params, nil)
			Convey("returns a VOD object", func() {
				So(err, ShouldBeNil)
				So(*vod.Title, ShouldEqual, "Kibler - Karazhan Week Three!")
			})
		})
	})
}

func TestGetVodByIDs(t *testing.T) {
	Convey("When calling GetVodsByIDs", t, func() {
		Convey("with valid ids", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, multiVodReponse)
			}))

			client, err := NewClient(twitchhttp.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			vods, err := client.GetVodsByIDs(context.Background(), []string{"3", "22"}, nil, nil)
			Convey("should return multiple vods", func() {
				So(err, ShouldBeNil)
				So(len(vods), ShouldEqual, 2)
			})
		})
		Convey("with an invalid id", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, errorResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			_, err = client.GetVodByID(context.Background(), "322", nil, nil)
			Convey("should cause an error", func() {
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestModeration(t *testing.T) {
	Convey("When calling Moderation", t, func() {
		Convey("with a valid id", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, nil)
			}))

			client, err := NewClient(twitchhttp.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			err = client.Moderation(context.Background(), "322", "1", nil)
			Convey("should return a success with no error", func() {
				So(err, ShouldBeNil)
			})
		})
	})
}

func TestGetVodsAggregationByIDs(t *testing.T) {
	Convey("When calling GetVodsAggregationByIDs", t, func() {
		Convey("when the service responds with OK status", func() {
			// mock a valid response
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, vodsAggregationResponse)
			}))
			client, _ := NewClient(twitchhttp.ClientConf{Host: ts.URL})

			vodsAggregation, err := client.GetVodsAggregationByIDs(context.Background(), []string{"322"}, nil, nil)
			Convey("returns a VOD aggregation object", func() {
				So(err, ShouldBeNil)
				So(vodsAggregation.TotalDuration, ShouldEqual, uint64(45))
			})
		})
		Convey("when the service responds with an error", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				w.WriteHeader(http.StatusInternalServerError)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			vodsAggregation, err := client.GetVodsAggregationByIDs(context.Background(), []string{"322"}, nil, nil)
			Convey("returns an error", func() {
				So(err, ShouldNotBeNil)
				So(vodsAggregation, ShouldBeNil)
			})
		})
		Convey("when using GetVodParams and the service responds with OK status", func() {
			ts := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				fmt.Fprintln(w, vodsAggregationResponse)
			}))
			client, err := NewClient(twitchhttp.ClientConf{Host: ts.URL})
			if err != nil {
				t.Fatal(err)
			}

			params := NewGetVodParams()
			params.IncludeDeleted = true
			vodsAggregation, err := client.GetVodsAggregationByIDs(context.Background(), []string{"322"}, params, nil)
			Convey("returns a VOD aggregation object", func() {
				So(err, ShouldBeNil)
				So(vodsAggregation.TotalCount, ShouldEqual, uint64(2))
			})
		})
	})
}
