package clients

import (
	"errors"
	"log"

	"golang.org/x/net/context"

	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/web/audrey/client/audrey"
	"github.com/cactus/go-statsd-client/statsd"
)

var (
	audreyHost      = "audrey-host"
	audreyStatsName = "clients.audrey"
)

const (
	audreyMaxIdleConnsPerHost = 10
)

// Audrey is the channel feed service interface
type Audrey interface {
	CreatePost(ctx context.Context, channelID, userID, content string, opts *twitchhttp.ReqOpts) (*audrey.Post, error)
}

func init() {
	config.Register(map[string]string{
		audreyHost: "http://audrey.dev.us-west2.justin.tv/",
	})
}

// NewAudreyClient creates a new Audrey client
func NewAudreyClient(stats statsd.Statter) (Audrey, error) {
	audreyURL := config.Resolve(audreyHost)
	if audreyURL == "" {
		return nil, errors.New("Nonexistent audrey host")
	}

	client, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host:  audreyURL,
		Stats: stats,
		Transport: twitchhttp.TransportConf{
			MaxIdleConnsPerHost: audreyMaxIdleConnsPerHost,
		},
		TimingXactName: audreyStatsName,
	})
	if err != nil {
		return nil, err
	}

	log.Printf("Connecting to Audrey at %s", audreyURL)

	return &audrey.Client{
		Client: client,
	}, nil
}
