package clients

import (
	"log"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/cloudwatchlogs"
	"github.com/aws/aws-sdk-go/service/iam"
)

// CloudwatchLogStream a connection to a Cloudwatch log stream
// used for sending logs to cloudwatch
type CloudwatchLogStream struct {
	svc       *cloudwatchlogs.CloudWatchLogs
	group     string
	name      string
	nextToken *string
}

// NewCloudwatchLogStream instantiates a new connection to a
// cloudwatch log stream
func NewCloudwatchLogStream(group, name string) (*CloudwatchLogStream, error) {
	session, config := CreateAWSClientConfigs()
	iamAPI := iam.New(session, config)
	accountInfo, err := iamAPI.ListAccountAliases(&iam.ListAccountAliasesInput{})
	if err != nil {
		return nil, err
	}
	log.Printf("Connecting to CloudWatchLogs stream in %v (%s)", *accountInfo.AccountAliases[0], *config.Region)
	cwlogs := cloudwatchlogs.New(session, config)

	params := &cloudwatchlogs.DescribeLogStreamsInput{
		LogGroupName:        aws.String(group), // Required
		Descending:          aws.Bool(true),
		Limit:               aws.Int64(1),
		LogStreamNamePrefix: aws.String(name),
	}
	desc, err := cwlogs.DescribeLogStreams(params)
	if err != nil {
		return nil, err
	}

	var nextToken *string
	if len(desc.LogStreams) > 0 {
		nextToken = desc.LogStreams[0].UploadSequenceToken
	} else {
		_, err := cwlogs.CreateLogStream(&cloudwatchlogs.CreateLogStreamInput{
			LogGroupName:  aws.String(group),
			LogStreamName: aws.String(name),
		})
		if err != nil {
			return nil, err
		}
	}

	return &CloudwatchLogStream{
		svc:       cwlogs,
		group:     group,
		name:      name,
		nextToken: nextToken,
	}, nil
}

// Log sends log messages to the log stream
func (s *CloudwatchLogStream) Log(messages []string) error {
	if s == nil {
		return nil
	}

	group := s.group
	name := s.name
	milliTime := time.Now().UTC().UnixNano() / int64(time.Millisecond)

	logEvents := []*cloudwatchlogs.InputLogEvent{}
	for _, message := range messages {
		logEvents = append(logEvents, &cloudwatchlogs.InputLogEvent{
			Message:   aws.String(message),
			Timestamp: aws.Int64(milliTime),
		})
	}

	params := &cloudwatchlogs.PutLogEventsInput{
		LogEvents:     logEvents,
		LogGroupName:  aws.String(group),
		LogStreamName: aws.String(name),
		SequenceToken: s.nextToken,
	}
	resp, err := s.svc.PutLogEvents(params)
	if err != nil {
		return err
	}

	s.nextToken = resp.NextSequenceToken
	return nil
}
