package clients

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"strconv"
	"time"

	"github.com/cactus/go-statsd-client/statsd"

	"golang.org/x/net/context"

	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/vod/vinyl/models"
)

var (
	partnershipsHost      = "partnerships_host"
	partnershipsStatsName = "clients.partnerships"
)

const (
	timeout                         = 2 * time.Second
	partnershipsMaxIdleConnsPerHost = 10
)

// PartnershipsHost stores the host url for the Partnerships service
type PartnershipsHost struct {
	Client twitchhttp.Client
}

// Properties stores the properties that are returned by the Partnerships service
type Properties struct {
	UserID                  int               `json:"user_id"`
	PartnerProgram          models.NullBool   `json:"partner_program"`
	PartnerProgramJoinDate  models.NullTime   `json:"partner_program_join_date"`
	PartnerQualityTier      models.NullString `json:"partner_quality_tier"`
	PartnerType             models.NullString `json:"partner_type"`
	TrackInLiverail         models.NullBool   `json:"track_in_liverail"`
	TwitchLiverailID        models.NullInt64  `json:"twitch_liverail_id"`
	PartnerLiverailID       models.NullInt64  `json:"partner_liverail_id"`
	TeespringConnectEnabled models.NullBool   `json:"teespring_connect_enabled"`
}

func init() {
	config.Register(map[string]string{
		partnershipsHost: "http://partnerships.dev.us-west2.twitch.tv",
	})
}

// NewPartnershipsClient creates a new Partnerships client and tries to get the
// host url with config
func NewPartnershipsClient(stats statsd.Statter) (*PartnershipsHost, error) {
	partnershipsURL := config.Resolve(partnershipsHost)
	if partnershipsURL == "" {
		return nil, errors.New("Nonexistent partnerships host")
	}

	client, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host:  partnershipsURL,
		Stats: stats,
		Transport: twitchhttp.TransportConf{
			MaxIdleConnsPerHost: partnershipsMaxIdleConnsPerHost,
		},
		TimingXactName: partnershipsStatsName,
	})
	if err != nil {
		return nil, err
	}

	return &PartnershipsHost{
		Client: client,
	}, nil
}

// GetPartnerProperties makes a request to the Partnerships service at the url
// in Partnerships and returns a struct with the properties
func (p *PartnershipsHost) GetPartnerProperties(ctx context.Context, userID int64) (Properties, error) {
	var result Properties

	request, err := p.Client.NewRequest("GET", "/partners/"+strconv.FormatInt(userID, 10)+"/properties", nil)
	if err != nil {
		return result, err
	}

	reqOpts := twitchhttp.ReqOpts{
		StatName:       fmt.Sprintf("%s.get_partner_properties", partnershipsStatsName),
		StatSampleRate: 1.0,
	}
	response, err := p.Client.Do(ctx, request, reqOpts)
	if err != nil {
		return result, err
	}
	defer func() {
		err := response.Body.Close()
		if err != nil {
			log.Print(err)
		}
	}()

	body, err := ioutil.ReadAll(response.Body)
	if response.StatusCode >= 300 {
		if err != nil {
			return result, fmt.Errorf("Could not parse error response: %s", err.Error())
		}
		return result, fmt.Errorf("Received status code: %d, %s", response.StatusCode, string(body))
	}

	err = json.Unmarshal(body, &result)
	if err != nil {
		return result, fmt.Errorf("Could not parse response: %s", err.Error())
	}
	return result, nil
}
