package clients

import (
	"time"

	"code.justin.tv/chat/db"
	"code.justin.tv/vod/vinyl/utils"
)

// Constants related to postgres variables and sandstorm keys for postgres login information.
var (
	dbDriverName       = "dbDriverName"
	maxOpenConnections = "maxOpenConnections"
	maxIdleConnections = "maxIdleConnections"
	maxQueueSize       = "maxQueueSize"
	connAcquireTimeout = "connRequestTimeout"
	requestTimeout     = "requestTimeout"
	maxConnectionAge   = "maxConnectionAge"
	VinylDBUser        = "vinyl_db_user_name"
	VinylDBPassword    = "vinyl_db_user_password"
)

// NewRdsReaderConf returns the database connection options for a read slave
func NewRdsReaderConf(dbHostName string, dbHostPort int, dbName string) []db.Option {
	return []db.Option{
		db.Host(dbHostName),
		db.Port(dbHostPort),
		db.DBName(dbName),
		db.DriverName("postgres"),
		db.MaxOpenConns(200),
		db.MaxIdleConns(200),
		db.MaxQueueSize(2000),
		db.ConnAcquireTimeout(3 * time.Second),
		db.RequestTimeout(30 * time.Second),
		db.MaxConnAge(300 * time.Second),
	}
}

// NewRdsMasterReaderConf returns the database connection options for the read master
func NewRdsMasterReaderConf(dbHostName string, dbHostPort int, dbName string) []db.Option {
	return []db.Option{
		db.Host(dbHostName),
		db.Port(dbHostPort),
		db.DBName(dbName),
		db.DriverName("postgres"),
		db.MaxOpenConns(50),
		db.MaxIdleConns(50),
		db.MaxQueueSize(2000),
		db.ConnAcquireTimeout(3 * time.Second),
		db.RequestTimeout(30 * time.Second),
		db.MaxConnAge(300 * time.Second),
	}
}

// NewRdsWriterConf returns the database connection options for the write master
func NewRdsWriterConf(dbHostName string, dbHostPort int, dbName string) []db.Option {
	return []db.Option{
		db.Host(dbHostName),
		db.Port(dbHostPort),
		db.DBName(dbName),
		db.DriverName("postgres"),
		db.MaxOpenConns(50),
		db.MaxIdleConns(50),
		db.MaxQueueSize(2000),
		db.ConnAcquireTimeout(3 * time.Second),
		db.RequestTimeout(30 * time.Second),
		db.MaxConnAge(300 * time.Second),
	}
}

// NewPostgresConn creates a postgres connection with the supplied credentials.
func NewPostgresConn(dbUser, dbPassword string, dbConf []db.Option) (db.DB, error) {
	user, err := utils.GetSecret(dbUser)
	if err != nil {
		return nil, err
	}

	password, err := utils.GetSecret(dbPassword)
	if err != nil {
		return nil, err
	}

	allOptions := append([]db.Option{
		db.User(user),
		db.Password(password)}, dbConf...)

	conn, err := db.Open(allOptions...)

	if err != nil {
		return nil, err
	}

	return conn, nil
}
