package clients

import (
	"encoding/json"
	"fmt"
	"log"

	"code.justin.tv/common/config"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
)

var (
	pushyTopicARN                       = "pushy-topic-arn"
	pushyStatsName                      = "clients.pushy"
	pushyVodUploadEvent                 = "vod_upload_complete"
	pushyModerationEvent                = "vod_moderation_complete"
	pushyEmailTypeFollowersNotification = "followers"
)

const (
	pushyStatSampleRate = 1.0
)

// Pushy is the interface to the pushy service used for email notifications
type Pushy interface {
	SendVodUploadNotification(ctx context.Context, channelID int64, vodID int64) error
	SendModerationEmail(ctx context.Context, channelID int64, vodID int64, emailType string) error
}

type pushyImpl struct {
	snsClient *sns.SNS
	stats     statsd.Statter
	topicARN  string
}

func init() {
	config.Register(map[string]string{
		pushyTopicARN: "arn:aws:sns:us-west-2:465369119046:vinyl_staging_vodupload_notifications",
	})
}

// NewPushyClient initializes and returns a new Pushy client
func NewPushyClient(stats statsd.Statter) (Pushy, error) {
	region := config.Resolve(AWSRegionName)
	svc := sns.New(session.New(), aws.NewConfig().WithRegion(region))

	return &pushyImpl{
		snsClient: svc,
		stats:     stats,
		topicARN:  config.Resolve(pushyTopicARN),
	}, nil
}

func (p *pushyImpl) SendVodUploadNotification(ctx context.Context, channelID int64, vodID int64) error {
	err := p.stats.Inc(fmt.Sprintf("%s.send_vod_upload_notification", pushyStatsName), 1, pushyStatSampleRate)
	if err != nil {
		log.Printf("error sending send_vod_upload_notification counter")
	}

	messageData := map[string]interface{}{
		"channel_id": channelID,
		"vod_id":     vodID,
		"email_type": pushyEmailTypeFollowersNotification,
	}
	message, err := json.Marshal(messageData)
	if err != nil {
		return err
	}

	params := &sns.PublishInput{
		Message: aws.String(string(message)),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				DataType:    aws.String("String"),
				StringValue: aws.String(pushyVodUploadEvent),
			},
		},
		TopicArn: &p.topicARN,
	}

	_, err = p.snsClient.Publish(params)
	if err != nil {
		return err
	}

	return nil
}

func (p *pushyImpl) SendModerationEmail(ctx context.Context, channelID int64, vodID int64, emailType string) error {
	err := p.stats.Inc(fmt.Sprintf("%s.send_moderation_email", pushyStatsName), 1, pushyStatSampleRate)
	if err != nil {
		log.Printf("error sending send_moderation_email counter")
	}

	messageData := map[string]interface{}{
		"channel_id": channelID,
		"vod_id":     vodID,
		"email_type": emailType,
	}
	message, err := json.Marshal(messageData)
	if err != nil {
		return err
	}

	params := &sns.PublishInput{
		Message: aws.String(string(message)),
		MessageAttributes: map[string]*sns.MessageAttributeValue{
			"event": {
				DataType:    aws.String("String"),
				StringValue: aws.String(pushyModerationEvent),
			},
		},
		TopicArn: &p.topicARN,
	}

	_, err = p.snsClient.Publish(params)
	if err != nil {
		return err
	}

	return nil
}
