package clients

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"

	"github.com/cactus/go-statsd-client/statsd"

	"golang.org/x/net/context"

	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
)

var (
	internalRailsHost        = "http://api.internal.twitch.tv"
	railsMaxIdleConnsPerHost = 10
	railsStatName            = "clients.rails"
)

// RailsHost stores the host url for the Rails service
type RailsHost struct {
	Client twitchhttp.Client
}

// NewRailsClient creates a new rails client and tries to get the
// host url with config
func NewRailsClient(stats statsd.Statter) (*RailsHost, error) {
	client, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host:  internalRailsHost,
		Stats: stats,
		Transport: twitchhttp.TransportConf{
			MaxIdleConnsPerHost: railsMaxIdleConnsPerHost,
		},
		TimingXactName: railsStatName,
	})
	if err != nil {
		return nil, err
	}

	return &RailsHost{
		Client: client,
	}, nil
}

// SendEmail makes a request to an internal Rails api to send an email with the appropriate arguments.
func (r *RailsHost) SendEmail(ctx context.Context, userID int64, emailName string, emailArgs interface{}) error {
	if config.Environment() != "production" {
		return nil // Never send this except for production.
	}

	emailBody := map[string]interface{}{
		"email_name": emailName,
		"user_id":    userID,
		"email_args": emailArgs,
	}
	body, err := json.Marshal(emailBody)
	if err != nil {
		return err
	}

	req, err := r.Client.NewRequest("POST", "/api/internal/emails", bytes.NewBuffer(body))
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/json")

	reqOpts := twitchhttp.ReqOpts{
		StatName:       fmt.Sprintf("%s.send_email", railsStatName),
		StatSampleRate: 1.0,
	}
	resp, err := r.Client.Do(context.Background(), req, reqOpts)
	if err != nil {
		return err
	}

	defer Close(resp.Body)

	if resp.StatusCode >= 300 {
		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return err
		}
		return errors.New(string(body))
	}
	return nil
}
