package clients

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"strconv"
	"strings"

	"github.com/cactus/go-statsd-client/statsd"

	"golang.org/x/net/context"

	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/vod/vinyl/models"
)

var (
	usersServiceHost = "users_service_host"
	usersStatsName   = "clients.users_service"
)

const (
	usersServiceMaxIdleConnsPerHost = 10
)

// UsersServiceClient stores the host url for the Users service
type UsersServiceClient struct {
	Client twitchhttp.Client
}

// UserInfo stores the properties that are returned by the users service
// https://git-aws.internal.justin.tv/web/users-service/blob/master/models/properties.go
type UserInfo struct {
	UserID                  int64             `json:"id"`
	Login                   models.NullString `json:"login"`
	DmcaViolation           models.NullBool   `json:"dmca_violation"`
	TermsOfServiceViolation models.NullBool   `json:"terms_of_service_violation"`
	DeletedOn               models.NullTime   `json:"deleted_on"`
	Language                models.NullString `json:"language"`
	Category                models.NullString `json:"category"`
	RemoteIP                models.NullString `json:"remote_ip"`
	Email                   models.NullString `json:"email"`
	LastLogin               models.NullString `json:"last_login"`
	BannedUntil             models.NullTime   `json:"banned_until"`
	DMCAViolationCount      models.NullInt64  `json:"dmca_violation_count"`
	TOSViolationCount       models.NullInt64  `json:"tos_violation_count"`
	DisplayName             models.NullString `json:"displayname"`
	EmailVerified           models.NullBool   `json:"email_verified"`
}

func init() {
	config.Register(map[string]string{
		usersServiceHost: "http://users-service.dev.us-west2.twitch.tv/",
	})
}

// NewUsersServiceClient creates a new Users Service client and tries to get the
// host url with config
func NewUsersServiceClient(stats statsd.Statter) (*UsersServiceClient, error) {
	usersServiceURL := config.Resolve(usersServiceHost)
	if usersServiceURL == "" {
		return nil, errors.New("Nonexistent users service host")
	}

	client, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host:  usersServiceURL,
		Stats: stats,
		Transport: twitchhttp.TransportConf{
			MaxIdleConnsPerHost: usersServiceMaxIdleConnsPerHost,
		},
		TimingXactName: usersStatsName,
	})
	if err != nil {
		return nil, err
	}

	log.Printf("Connecting to Users service at %s", usersServiceURL)

	return &UsersServiceClient{
		Client: client,
	}, nil
}

// GetUserInfoBatch fetches user info for multiple user IDs.
func (u *UsersServiceClient) GetUserInfoBatch(ctx context.Context, userIDs []int64) ([]*UserInfo, error) {
	stringIDs := []string{}
	for _, id := range userIDs {
		stringIDs = append(stringIDs, strconv.FormatInt(id, 10))
	}
	url := fmt.Sprintf("/users?id=%v", strings.Join(stringIDs, "&id="))
	req, err := u.Client.NewRequest("GET", url, nil)
	if err != nil {
		return []*UserInfo{}, err
	}

	reqOpts := twitchhttp.ReqOpts{
		StatName:       fmt.Sprintf("%s.get_user_info_batch", usersStatsName),
		StatSampleRate: 1.0,
	}
	resp, err := u.Client.Do(context.Background(), req, reqOpts)
	if err != nil {
		return []*UserInfo{}, err
	}

	defer Close(resp.Body)

	body, err := ioutil.ReadAll(resp.Body)
	if resp.StatusCode >= 300 {
		if err != nil {
			return []*UserInfo{}, fmt.Errorf("Could not parse error response: %s", err.Error())
		}
		return []*UserInfo{}, fmt.Errorf("Received status code: %d, %s", resp.StatusCode, string(body))
	}

	var result map[string][]*UserInfo
	err = json.Unmarshal(body, &result)
	if err != nil {
		return []*UserInfo{}, fmt.Errorf("Could not parse response: %s", err.Error())
	}

	return result["results"], nil
}

// GetUserInfo fetches user info for a specific user ID.
func (u *UsersServiceClient) GetUserInfo(ctx context.Context, userID int64) (UserInfo, error) {
	var result UserInfo
	req, err := u.Client.NewRequest("GET", fmt.Sprintf("/users/%d", userID), nil)
	if err != nil {
		return result, err
	}

	reqOpts := twitchhttp.ReqOpts{
		StatName:       fmt.Sprintf("%s.get_user_info", usersStatsName),
		StatSampleRate: 1.0,
	}
	resp, err := u.Client.Do(context.Background(), req, reqOpts)
	if err != nil {
		return result, err
	}

	defer Close(resp.Body)

	body, err := ioutil.ReadAll(resp.Body)
	if resp.StatusCode >= 300 {
		if err != nil {
			return result, fmt.Errorf("Could not parse error response: %s", err.Error())
		}
		return result, fmt.Errorf("Received status code: %d, %s", resp.StatusCode, string(body))
	}

	err = json.Unmarshal(body, &result)
	if err != nil {
		return result, fmt.Errorf("Could not parse response: %s", err.Error())
	}

	return result, nil
}
