package clients

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"strconv"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"

	connections "code.justin.tv/identity/connections/client"
)

const rabbitVodsHost = "http://usher.justin.tv/"

// Usher is a client for making requests to Usher
type Usher struct {
	rabbit twitchhttp.Client
}

// NewUsherClient instantiates an usher client
func NewUsherClient() *Usher {
	rabbitClient, err := twitchhttp.NewClient(twitchhttp.ClientConf{
		Host: rabbitVodsHost,
	})
	if err != nil {
		return nil
	}
	return &Usher{
		rabbit: rabbitClient,
	}
}

// DeleteVods tells usher to bulk delete vods
func (u *Usher) DeleteVods(ids []int64) error {
	route := "vod.delete"
	statName := "bulk_delete"
	body := map[string][]int64{"ids": ids}
	return u.PostJob(route, body, statName)
}

// UnmuteAudio sends a job to usher to unmute the audio that has been appealed.
func (u *Usher) UnmuteAudio(amrID, vodID int64) error {
	route := "vod.audio_unmute_filter"
	body := map[string]int64{
		"amr_id": amrID,
		"vod_id": vodID,
	}
	statName := "unmute_audio"
	return u.PostJob(route, body, statName)
}

// CreateHighlight sends a request to start the creation of a new highlight
func (u *Usher) CreateHighlight(archiveID, highlightID int64) error {
	route := "vod.highlight"
	statName := "create_highlight"
	highlightBody := map[string]int64{
		"id":           archiveID,
		"highlight_id": highlightID,
	}
	return u.PostJob(route, highlightBody, statName)

}

// Rebroadcast sends a request to rebroadcast a vod
func (u *Usher) Rebroadcast(vodIDs []int64, streamKey string) error {
	route := "vod.rebroadcast"
	statName := "rebroadcast"
	body := map[string]interface{}{
		"vod_ids":    vodIDs,
		"stream_key": streamKey,
	}
	return u.PostJob(route, body, statName)
}

// YoutubeExport sends a request to export a vod to Youtube
func (u *Usher) YoutubeExport(input *models.YoutubeExportInput, ytUser *connections.YoutubeUser, login string) error {
	description := input.Description
	if description == "" {
		description = "Broadcasted live on Twitch"
	}
	url := fmt.Sprintf("https://www.twitch.tv/%s", login)

	privacySettings := "public"
	if input.Private {
		privacySettings = "private"
	}

	route := "vod.export.youtube"
	statName := "youtube_export"
	body := map[string]interface{}{
		"vod_id":           input.VodID,
		"token":            "www.twitch.tv",
		"auth_token":       ytUser.Token,
		"auth_type":        "OAuth2",
		"description":      fmt.Sprintf("%s -- Watch live at %s", description, url),
		"title":            input.Title,
		"keywords":         fmt.Sprintf("twitch, games, %s", input.TagList),
		"category":         "Games",
		"privacy_settings": privacySettings,
		"do_split":         strconv.FormatBool(input.DoSplit),
		"refresh_token":    ytUser.RefreshToken,
	}
	return u.PostJob(route, body, statName)
}

// PostJob sends a job to usher with the specified route and body.
func (u *Usher) PostJob(route string, jobBody interface{}, statName string) error {
	body, err := json.Marshal(jobBody)
	if err != nil {
		return err
	}

	req, err := u.rabbit.NewRequest("POST", fmt.Sprintf("/job/send/%s", route), bytes.NewBuffer(body))
	if err != nil {
		return err
	}
	req.Header.Add("Content-Type", "application/json")

	resp, err := u.rabbit.Do(context.Background(), req, twitchhttp.ReqOpts{StatName: statName})
	if err != nil {
		return err
	}
	defer Close(resp.Body)

	if resp.StatusCode >= 300 {
		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return err
		}
		return errors.New(string(body))
	}

	return nil
}

// Close closes what it is given
func Close(c io.Closer) {
	err := c.Close()
	if err != nil {
	}
}
