package datastore

import (
	"time"

	dbmodels "code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	"code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

// Named is implemented by objects that have a name.
type Named interface {
	Name() string
}

// HealthReporter is implemented by objects that can report health and utilization.
type HealthReporter interface {
	Health(context.Context) error
	Utilization(context.Context) float32
}

// Writer is implemented by objects that write/update entries to the database.
type Writer interface {
	CreateAMRs(context.Context, int64, []*models.AMR, bool) ([]int64, error)
	CreateHighlight(context.Context, *models.HighlightInput, bool) (*dbmodels.Vod, error)
	CreatePastBroadcast(context.Context, *models.Vod, bool) (*dbmodels.Vod, error)
	CreateThumbnails(context.Context, int64, models.Thumbnails) (models.Thumbnails, error)
	CreateTrackAppeals(context.Context, []*models.TrackAppeal, int64, bool) ([]int64, error)
	CreateVodAppeal(context.Context, *models.VodAppeal, bool) (int64, error)
	DeleteThumbnail(context.Context, int64, string) error
	DeleteThumbnails(context.Context, int64) error
	DeleteVods(context.Context, []int64) error
	ResolveTrackAppeal(context.Context, int64, bool) (int64, int64, error)
	ResolveVodAppeal(context.Context, int64) (int64, error)
	SetUserVideoPrivacyProperties(context.Context, int64, *models.UserVideoPrivacyPropertiesInput) (*models.UserVideoPrivacyProperties, error)
	SetUserVODProperties(context.Context, int64, *models.UserVODPropertiesInput) (*dbmodels.UserVodProperties, error)
	SetViewcounts(context.Context, map[int]int64) ([]*dbmodels.Vod, error)
	UpdateAMR(context.Context, int, map[string]interface{}) (*dbmodels.AMR, error)
	UpdateSelectedThumbnail(context.Context, int64, string) error
	UpdateVod(context.Context, int64, *models.VodUpdateInput) (*dbmodels.Vod, error)
	UpdateVodNotificationSettings(context.Context, int64, string, *models.VodNotificationSettingsInput) (*dbmodels.VodNotificationSettings, error)
}

// NamedWriter combines the Writer, Named and HealthReporter interfaces.
type NamedWriter interface {
	Named
	HealthReporter
	Writer
}

// Reader is implemented by objects that reads entries from the database.
type Reader interface {
	GetStaleUnmoderatedVods(context.Context) ([]*dbmodels.Vod, error)
	GetAMRs(context.Context, int) ([]*dbmodels.AMR, error)
	GetAMRsByVodIDs(context.Context, []int64) ([]*dbmodels.AMR, error)
	GetIntervalVodIDsToDelete(context.Context, time.Time, time.Time) ([]int64, error)
	GetRelatedVods(context.Context, int, int) ([]*dbmodels.Vod, error)
	GetScheduledVods(context.Context) ([]*dbmodels.Vod, error)
	GetTopVods(context.Context, []string, []string, string, string, string, int, int) ([]*dbmodels.Vod, error)
	GetTrackAppealsByVodAppealIDs(context.Context, []int64) ([]*dbmodels.TrackAppeal, error)
	GetTranscodingVodsCountByUser(context.Context, int64) (int64, error)
	GetUserVideoPrivacyProperties(context.Context, int64) (*models.UserVideoPrivacyProperties, error)
	GetUserVideoPrivacyPropertiesBatch(context.Context, []int64) (map[int64]models.UserVideoPrivacyProperties, error)
	GetUserVODProperties(context.Context, int64) (*dbmodels.UserVodProperties, error)
	GetVodAppeals(context.Context, models.NullBool, models.NullBool, string, models.NullInt64, int, int) ([]*dbmodels.VodAppeal, error)
	GetVodAppealsCount(context.Context, models.NullBool, models.NullBool, string, models.NullInt64) (int64, error)
	GetVodAppealsByVodIDs(context.Context, []int64) ([]*dbmodels.VodAppeal, error)
	GetVodIDsByUser(context.Context, int64) ([]int64, error)
	GetVodNotificationSettingsByVodIDs(context.Context, []int64) ([]*dbmodels.VodNotificationSettings, error)
	GetVodThumbnails(context.Context, int64) (models.Thumbnails, error)
	GetVodsByID(context.Context, []int64, *models.VodFilters) ([]*dbmodels.Vod, error)
	GetVodsByStatus(context.Context, string, []string, time.Time, time.Time) ([]*dbmodels.Vod, error)
	GetVodsByUser(context.Context, int, []string, []string, []string, *models.VodFilters, string, int, int) ([]*dbmodels.Vod, error)
	GetVodsCountByUser(context.Context, int, []string, []string, []string, *models.VodFilters) (int64, error)
	GetVodsAggregationByID(context.Context, []int64, *models.VodFilters) (*dbmodels.VodsAggregation, error)
	FollowedVods(context.Context, []int, []string, []string, []string, string, int, int) ([]*dbmodels.Vod, error)
	OwnerForRechat(ctx context.Context, vodID int64) (*dbmodels.VodOwner, error)
}

// NamedReader combines the Reader, Named and HealthReporter interfaces.
type NamedReader interface {
	Named
	HealthReporter
	Reader
}
