package vinyldb

import (
	"strings"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/chitin/clog"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	edgemodels "code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

// CreatePastBroadcast calls VinylDB to insert the past broadcast entry.
// It checks for the existence of the past broadcast by using URI
// and merely returns that if it already exists.
func (b *Backend) CreatePastBroadcast(ctx context.Context, vod *edgemodels.Vod, forceID bool) (*models.Vod, error) {
	// We first check uniqueness on the vod to see if any exist already.
	// This will be removed once we have completed the migration and we are able to use our own db with
	// upsert functionality.
	if len(vod.URI) > 0 {
		query := db.BuildQuery(
			models.FetchAllVODFieldsQuery(),
			"WHERE uri = ", db.Param)

		rows, err := b.conn.Query(ctx, "get_vod_by_uri", query, vod.URI)
		existingVods, err := models.ReadVodRows(rows, err, b.logger)
		if err != nil {
			return nil, err
		}

		if len(existingVods) > 0 {
			err = b.AttachVodThumbnails(ctx, existingVods)
			if err != nil {
				return nil, err
			}
			return existingVods[0], nil
		}
	}

	dVod, err := models.FromVinylVod(vod)
	if err != nil {
		return nil, err
	}

	paramNames := models.VodCreateFields
	valuesList := dVod.ValuesList()
	if forceID {
		paramNames = append(paramNames, "id")
		valuesList = append(valuesList, vod.ID)
	}

	valueParams := BuildValuesList(1, len(paramNames))

	query := db.BuildQuery(
		"INSERT INTO", models.VodTableName,
		BuildParamList(paramNames),
		"VALUES", valueParams,
		"RETURNING", strings.Join(models.VodFields, ","))

	clog.Log(ctx).Printf("writer=%s query=%q", b.Name(), query)
	rows, err := b.conn.Query(ctx, "create_vod", query, valuesList...)

	createdVods, err := models.ReadVodRows(rows, err, b.logger)
	if err != nil {
		return nil, err
	}
	createdVod := createdVods[0]

	createdVinylVod, err := createdVod.AsVinylVod()
	if err != nil {
		return nil, err
	}

	if len(vod.Thumbnails) > 0 {
		_, err = b.CreateThumbnails(ctx, createdVinylVod.ID, vod.Thumbnails)
		if err != nil {
			return nil, err
		}
	}

	completeCreatedVods, err := b.GetVodsByID(ctx, []int64{createdVinylVod.ID}, edgemodels.VodFilterNone())
	if err != nil {
		return nil, err
	}
	completeCreatedVod := completeCreatedVods[0]

	return completeCreatedVod, nil
}
