package vinyldb

import (
	"regexp"
	"testing"
	"time"

	sqlmock "github.com/DATA-DOG/go-sqlmock"

	. "github.com/smartystreets/goconvey/convey"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	edgemodels "code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

func TestCreatePastBroadcast(t *testing.T) {
	Convey("Creating broadcasts", t, func() {
		mock, backend := Setup()

		Convey("Inserts values and runs with 1 vod", func() {
			now := time.Now().Round(time.Second)
			vod := &edgemodels.Vod{
				URI:           "hello",
				BroadcastID:   34,
				Status:        edgemodels.StatusRecording,
				OwnerID:       124,
				BroadcastType: "archive",
				Offset:        0,
				Views:         0,
				Duration:      120349,
				CreatedAt:     now,
				UpdatedAt:     now,
				StartedOn:     now,
			}

			query1 := models.FetchAllVODFieldsQuery() + ` WHERE uri = $1`
			mock.ExpectQuery(regexp.QuoteMeta(query1)).WithArgs("hello").WillReturnRows(sqlmock.NewRows([]string{}))

			query2 := `INSERT INTO vods ` +
				`(bitrates,broadcast_id,broadcast_type,broadcaster_software,` +
				`codecs,created_at,created_by_id,deleted,delete_at,` +
				`description,description_html,display_names,duration,formats,fps,` +
				`game,language,manifest,"offset",owner_id,origin,` +
				`playlist_preferences,published_at,resolutions,source_archive_id,started_on,` +
				`status,tag_hstore,title,updated_at,uri,views_count,viewable_at,viewable,selected_thumbnail_id) ` +
				`VALUES ($1,$2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,$13,$14,$15,$16,` +
				`$17,$18,$19,$20,$21,$22,$23,$24,$25,$26,$27,$28,$29,$30,$31,$32,$33,$34,$35) ` +
				`RETURNING ` +
				`id,bitrates,broadcast_id,broadcast_type,broadcaster_software,` +
				`codecs,created_at,created_by_id,deleted,delete_at,` +
				`description,description_html,display_names,duration,formats,fps,` +
				`game,language,manifest,"offset",owner_id,origin,` +
				`playlist_preferences,published_at,resolutions,source_archive_id,started_on,` +
				`status,tag_hstore,title,updated_at,uri,views_count,viewable_at,viewable,selected_thumbnail_id`
			fakeResult := sqlmock.NewRows(models.VodFields)
			fakeResult.AddRow(
				1, nil, 34, "archive", nil, nil,
				now, nil, nil, nil, nil, nil, nil,
				1238, nil, nil, nil, nil, nil,
				0, 111, nil, nil, nil, nil, nil,
				now, edgemodels.StatusRecording, nil, nil, now,
				"hello", 0, nil, "public", nil)
			mock.ExpectQuery(regexp.QuoteMeta(query2)).WithArgs(
				nil, 34, "archive", nil, nil, now,
				nil, nil, nil, nil, nil, nil, 120349,
				nil, nil, nil, nil, nil, 0,
				124, nil, nil, nil, nil, nil, now,
				edgemodels.StatusRecording, nil, nil, now, "hello",
				0, nil, "public", nil).WillReturnRows(fakeResult)

			fakeGet := sqlmock.NewRows(models.VodFields)
			fakeGet.AddRow(
				1, nil, 34, "archive", nil, nil,
				now, nil, nil, nil, nil, nil, nil,
				1238, nil, nil, nil, nil, nil,
				0, 111, nil, nil, nil, nil, nil,
				now, edgemodels.StatusRecording, nil, nil, now,
				"hello", 0, nil, "", nil)
			query3 := `SELECT id,bitrates,broadcast_id,broadcast_type,broadcaster_software,codecs,created_at,created_by_id,deleted,delete_at,description,description_html,display_names,duration,formats,fps,game,language,manifest,"offset",owner_id,origin,playlist_preferences,published_at,resolutions,source_archive_id,started_on,status,tag_hstore,title,updated_at,uri,views_count,viewable_at,viewable,selected_thumbnail_id FROM vods WHERE id IN ($1)`
			mock.ExpectQuery(regexp.QuoteMeta(query3)).WillReturnRows(fakeGet)

			fakeVodThumbnails := sqlmock.NewRows([]string{"id", "vod_id", "path", "offset", "type"})
			mock.ExpectQuery(regexp.QuoteMeta(`SELECT id, vod_id, path, "offset", type FROM vod_thumbnails WHERE vod_id IN ( 1 )`)).WillReturnRows(fakeVodThumbnails)

			_, err := backend.CreatePastBroadcast(context.Background(), vod, false)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

		Convey("Doesn't try to create VODs if all inputs are created already", func() {
			vod := &edgemodels.Vod{
				OwnerID:     111,
				BroadcastID: 34,
				URI:         "happy",
			}

			query1 := models.FetchAllVODFieldsQuery() + ` WHERE uri = $1`
			now := time.Now().Round(time.Second)
			fakeResult := sqlmock.NewRows(models.VodFields)
			fakeResult.AddRow(
				1, nil, 34, "archive", nil, nil,
				now, nil, nil, nil, nil, nil, nil,
				1238, nil, nil, nil, nil, nil,
				0, 111, nil, nil, nil, nil, nil,
				now, edgemodels.StatusRecording, nil, nil, now,
				"happy", 0, nil, "public", nil)
			mock.ExpectQuery(regexp.QuoteMeta(query1)).WithArgs("happy").WillReturnRows(fakeResult)

			fakeVodThumbnails := sqlmock.NewRows([]string{"id", "vod_id", "path", "offset", "type"})
			mock.ExpectQuery(regexp.QuoteMeta(`SELECT id, vod_id, path, "offset", type FROM vod_thumbnails WHERE vod_id IN ( 1 )`)).WillReturnRows(fakeVodThumbnails)

			_, err := backend.CreatePastBroadcast(context.Background(), vod, false)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

	})
}
