package vinyldb

import (
	"database/sql"
	"log"
	"strings"
	"time"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/chitin/clog"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	"code.justin.tv/vod/vinyl/errors"
	"code.justin.tv/vod/vinyl/markdown"
	edgemodels "code.justin.tv/vod/vinyl/models"
	"github.com/lib/pq"
	"golang.org/x/net/context"
)

// CreateHighlight calls VinylDB to insert the highlight entry.
// It queries for the original broadcast object first and throws
// an error if it doesn't exist.
func (b *Backend) CreateHighlight(ctx context.Context, highlightInfo *edgemodels.HighlightInput, forceID bool) (*models.Vod, error) {
	vodID := highlightInfo.VodID

	vods, err := b.GetVodsByID(ctx, []int64{vodID}, edgemodels.VodFilterDeleted())
	if err != nil {
		return nil, err
	}

	if len(vods) < 1 {
		return nil, errors.NotFoundError{Type: "VOD", ID: int(vodID)}
	}

	vod := vods[0]

	vod.CreatedAt = time.Now().UTC().Round(time.Second)
	vod.UpdatedAt = time.Now().UTC().Round(time.Second)
	vod.BroadcastType = "highlight"
	vod.Status = "unprocessed"
	vod.Views = 0
	vod.Manifest = sql.NullString{Valid: false}
	vod.Title = sql.NullString(highlightInfo.Title)
	vod.Description = sql.NullString(highlightInfo.Description)
	vod.DescriptionHTML = sql.NullString{Valid: true, String: markdown.ConvertMarkdown(vod.Description.String)}
	vod.Game = sql.NullString(highlightInfo.Game)
	vod.Language = sql.NullString(highlightInfo.Language)
	start := int(highlightInfo.StartSeconds)
	end := int(highlightInfo.EndSeconds)
	vod.Offset = start
	vod.Duration = end - start
	vod.StartedOn = vod.StartedOn.UTC().Add(time.Duration(start) * time.Second)
	vod.PublishedAt = pq.NullTime{Time: time.Now().UTC().Round(time.Second), Valid: true}

	vod.SourceArchiveID = sql.NullInt64{Valid: true, Int64: vodID}
	if highlightInfo.TagList != "" {
		vod.TagHstore, err = models.TagListToHStore(highlightInfo.TagList)
		if err != nil {
			return nil, err
		}
	}
	vod.CreatedBy = sql.NullInt64(highlightInfo.CreatedBy)

	thumbnails := vod.ThumbnailModels()
	vod.Thumbnails = models.VodThumbnails{}

	paramNames := models.VodCreateFields
	valuesList := vod.ValuesList()
	if forceID {
		paramNames = append(paramNames, "id")
		valuesList = append(valuesList, highlightInfo.HighlightID)
	}

	valueParams := BuildValuesList(1, len(paramNames))

	query := db.BuildQuery(
		"INSERT INTO", models.VodTableName,
		BuildParamList(paramNames),
		"VALUES", valueParams,
		"RETURNING", strings.Join(models.VodFields, ","))

	clog.Log(ctx).Printf("writer=%s query=%q", b.Name(), query)
	rows, err := b.conn.Query(ctx, "create_highlight", query, valuesList...)
	createdVods, err := models.ReadVodRows(rows, err, b.logger)
	if err != nil {
		return nil, err
	}
	err = b.AttachVodThumbnails(ctx, createdVods)
	if err != nil {
		return nil, err
	}
	createdVod := createdVods[0]

	if len(thumbnails) > 0 {
		_, err := b.CreateThumbnails(ctx, createdVod.ID, thumbnails)
		if err != nil {
			log.Println(err)
		}
	}

	return createdVod, nil
}
