package vinyldb

import (
	"database/sql"
	"database/sql/driver"
	"regexp"
	"testing"
	"time"

	sqlmock "github.com/DATA-DOG/go-sqlmock"

	. "github.com/smartystreets/goconvey/convey"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	"code.justin.tv/vod/vinyl/markdown"
	edgemodels "code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

func TestCreateHighlight(t *testing.T) {
	Convey("Creating highlights", t, func() {
		mock, backend := Setup()

		now := time.Now().Round(time.Second)
		fakeVod := &models.Vod{
			ID:                  1,
			BroadcastID:         34,
			BroadcastType:       "archive",
			BroadcasterSoftware: sql.NullString{Valid: false},
			CreatedAt:           now,
			OwnerID:             777,
			Deleted:             sql.NullBool{Valid: false},
			Description:         sql.NullString{Valid: false},
			DescriptionHTML:     sql.NullString{Valid: false},
			Duration:            1000,
			Fps:                 sql.NullString{Valid: false},
			Game:                sql.NullString{Valid: false},
			Language:            sql.NullString{Valid: false},
			Manifest:            sql.NullString{Valid: false},
			Offset:              0,
			Origin:              sql.NullString{Valid: false},
			Resolutions:         sql.NullString{Valid: false},
			StartedOn:           now,
			Status:              edgemodels.StatusRecording,
			Title:               sql.NullString{Valid: false},
			UpdatedAt:           now,
			URI:                 "thecowranfast",
			Views:               5764,
			Viewable:            sql.NullString{Valid: false},
		}

		Convey("Returns an error if the source vod is not found", func() {
			highlightInput := &edgemodels.HighlightInput{
				VodID:        fakeVod.ID,
				StartSeconds: 17,
				EndSeconds:   60,
			}

			query := `SELECT (.+) FROM vods WHERE id IN (.+)`

			args := []driver.Value{fakeVod.ID}
			for _, v := range fakeVod.ValuesList() {
				args = append(args, v)
			}
			fakeResult := sqlmock.NewRows(models.VodFields)
			mock.ExpectQuery(query).WithArgs(fakeVod.ID).WillReturnRows(fakeResult)

			_, err := backend.CreateHighlight(context.Background(), highlightInput, false)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err.Error(), ShouldContainSubstring, "VOD with ID 1 not found")
		})

		Convey("Return the created highlight if successful", func() {
			highlightInput := &edgemodels.HighlightInput{
				VodID:        fakeVod.ID,
				StartSeconds: 17,
				EndSeconds:   60,
				Description:  edgemodels.NullString{Valid: true, String: "i play games"},
			}

			query := `SELECT (.+) FROM vods WHERE id IN (.+)`
			fakeResult := sqlmock.NewRows(models.VodFields)

			fakeResult.AddRow(
				fakeVod.ID, nil, fakeVod.BroadcastID, fakeVod.BroadcastType,
				fakeVod.BroadcasterSoftware.String, nil, fakeVod.CreatedAt, fakeVod.CreatedBy.Int64,
				fakeVod.Deleted.Bool, fakeVod.DeleteAt.Time, fakeVod.Description.String, fakeVod.DescriptionHTML.String, nil,
				fakeVod.Duration, nil, fakeVod.Fps.String, fakeVod.Game.String, fakeVod.Language.String,
				fakeVod.Manifest.String, fakeVod.Offset, fakeVod.OwnerID, fakeVod.Origin.String,
				nil, fakeVod.PublishedAt, fakeVod.Resolutions.String, fakeVod.SourceArchiveID.Int64, fakeVod.StartedOn,
				fakeVod.Status, nil, fakeVod.Title.String, fakeVod.UpdatedAt,
				fakeVod.URI, fakeVod.Views, fakeVod.ViewableAt, nil, nil)
			mock.ExpectQuery(query).WithArgs(fakeVod.ID).WillReturnRows(fakeResult)

			fakeVodThumbnails := sqlmock.NewRows([]string{"id", "vod_id", "path", "offset", "type"})
			mock.ExpectQuery(regexp.QuoteMeta(`SELECT id, vod_id, path, "offset", type FROM vod_thumbnails WHERE vod_id IN ( 1 )`)).WillReturnRows(fakeVodThumbnails)

			insertResults := sqlmock.NewRows(models.VodFields)
			insertResults.AddRow(
				2, nil, fakeVod.BroadcastID, "highlight",
				nil, nil, time.Now(), fakeVod.OwnerID,
				nil, time.Now(), highlightInput.Description.String, nil, nil,
				43, nil, nil, nil, nil,
				nil, highlightInput.StartSeconds, fakeVod.OwnerID, nil,
				nil, time.Now(), nil, fakeVod.ID, time.Now(),
				"unprocessed", nil, nil, time.Now(),
				fakeVod.URI, 0, nil, nil, nil)

			insertQuery := `INSERT INTO vods (.+) VALUES (.+) RETURNING (.+)`
			mock.ExpectQuery(insertQuery).WithArgs(
				nil, fakeVod.BroadcastID, "highlight",
				fakeVod.BroadcasterSoftware.String, nil, AnyTime{}, nil,
				fakeVod.Deleted.Bool, AnyTime{}, highlightInput.Description.String, markdown.ConvertMarkdown(highlightInput.Description.String), nil,
				43, nil, fakeVod.Fps.String, nil, nil,
				sql.NullString{Valid: false}, highlightInput.StartSeconds, fakeVod.OwnerID, fakeVod.Origin.String,
				nil, AnyTime{}, fakeVod.Resolutions.String, fakeVod.ID, AnyTime{},
				"unprocessed", nil, nil, AnyTime{},
				fakeVod.URI, 0, nil, nil, nil,
			).WillReturnRows(insertResults)

			mock.ExpectQuery(regexp.QuoteMeta(`SELECT id, vod_id, path, "offset", type FROM vod_thumbnails WHERE vod_id IN ( 2 )`)).WillReturnRows(fakeVodThumbnails)

			vod, err := backend.CreateHighlight(context.Background(), highlightInput, false)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
			vinylVod, err := vod.AsVinylVod()
			So(err, ShouldBeNil)
			So(vinylVod.Description.String, ShouldEqual, highlightInput.Description.String)

		})
	})
}
