package vinyldb

import (
	"fmt"
	"strconv"
	"strings"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/golibs/errorlogger"
	"github.com/cactus/go-statsd-client/statsd"
	"golang.org/x/net/context"
)

var pgSortValues = map[string]string{
	"time":     "published_at DESC",
	"views":    "views_count DESC",
	"trending": "log(views_count + 1) + EXTRACT(EPOCH FROM started_on)/86400 DESC",
	"trending_v3": "log(views_count + 1) + (0.5 * EXTRACT(EPOCH FROM started_on)/86400) DESC",
}

// Backend is the representation of a backend connection to a database.
type Backend struct {
	name   string
	conn   db.DB
	stats  statsd.Statter
	logger errorlogger.ErrorLogger
}

// New creates a Backend object that contains the database connection.
func New(conn db.DB, name string, stats statsd.Statter, logger errorlogger.ErrorLogger) *Backend {
	return &Backend{
		name:   name,
		conn:   conn,
		stats:  stats,
		logger: logger,
	}
}

// Name returns the name of the database.
func (B *Backend) Name() string {
	return B.name
}

// Utilization returns the current % of max open connections that the database is using.
func (B *Backend) Utilization(ctx context.Context) float32 {
	return float32(B.conn.Info().MaxOpenConns) / float32(B.conn.Info().OpenConnsCap)
}

// Health currently returns nil.
func (B *Backend) Health(ctx context.Context) error {
	//	rows, err := B.query(ctx, "health", NewQuery("SELECT 1"))
	//	defer rows.Close()
	//	return err
	return nil
}

// BuildParamList returns a list of params encased on parenthese
// Used for inserting/updating SQL queries.
func BuildParamList(params []string) string {
	return "(" + strings.Join(params, ",") + ")"
}

// BuildValuesList creates a string that is filled with
// db.Params values starting from a certain number.
func BuildValuesList(starting, toFill int) string {
	valueStrings := []string{}
	for i := starting; i < starting+toFill; i++ {
		numString := "$" + strconv.Itoa(i)
		valueStrings = append(valueStrings, numString)
	}
	return "(" + strings.Join(valueStrings, ",") + ")"
}

func pgSortQueryPart(sort string) string {
	return fmt.Sprintf("ORDER BY %s", pgSortValues[sort])
}
