package vinyldb

import (
	"code.justin.tv/chat/db"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	"golang.org/x/net/context"
)

// GetRelatedVods calls vinylDB to find vods either from the same broadcast or user.
func (b *Backend) GetRelatedVods(ctx context.Context, userID, vodID int) ([]*models.Vod, error) {
	var err error
	res1 := []*models.Vod{}
	if vodID != 0 {
		res1, err = b.withSameBroadcastID(ctx, vodID)
		if err != nil {
			return nil, err
		}
	}

	res2, err := b.fromSameOwner(ctx, userID, vodID)
	if err != nil {
		return nil, err
	}

	vods := append(res1, res2...)
	err = b.AttachVodThumbnails(ctx, vods)
	if err != nil {
		return nil, err
	}
	return vods, nil
}

func (b *Backend) withSameBroadcastID(ctx context.Context, vodID int) ([]*models.Vod, error) {
	query := db.BuildQuery(models.FetchAllVODFieldsQuery(),
		"WHERE broadcast_id = (SELECT broadcast_id FROM vods WHERE id =", db.Param, ")",
		"AND id !=", db.Param,
		models.Watchable(),
		"ORDER BY created_at DESC LIMIT 5",
	)
	rows, err := b.conn.Query(ctx, "related.with_same_broadcast_id", query, vodID, vodID)
	return models.ReadVodRows(rows, err, b.logger)
}

func (b *Backend) fromSameOwner(ctx context.Context, userID int, vodID int) ([]*models.Vod, error) {
	params := []interface{}{userID}
	queryParts := []interface{}{models.FetchAllVODFieldsQuery(), "WHERE owner_id =", db.Param}
	if vodID != 0 {
		queryParts = append(queryParts, "AND broadcast_id != (SELECT broadcast_id FROM vods WHERE id =", db.Param, ")")
		params = append(params, vodID)
	}
	queryParts = append(queryParts, models.Watchable(), "ORDER BY created_at DESC LIMIT 10")

	rows, err := b.conn.Query(ctx, "related.from_same_owner", db.BuildQuery(queryParts...), params...)
	return models.ReadVodRows(rows, err, b.logger)
}
