package vinyldb

import (
	"database/sql"
	"regexp"
	"testing"
	"time"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	edgemodels "code.justin.tv/vod/vinyl/models"

	"github.com/DATA-DOG/go-sqlmock"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func TestGetUserVideoPrivacyPropertiesBatch(t *testing.T) {
	now := time.Now()
	Convey("Fetching props for a user", t, func() {
		mock, reader := Setup()

		Convey("Fetches a user's props if it exists", func() {
			userIDs := []int64{ int64(123), int64(124) }
			query := `SELECT user_id,hide_archives,created_at,updated_at FROM user_video_privacy_properties WHERE user_id IN ($1, $2)`
			fakeResult := sqlmock.NewRows(models.UserVideoPrivacyPropertiesFields)
			fakeResult.AddRow(123, false, now, now)
			fakeResult.AddRow(124, true, now, now)
			mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(userIDs[0], userIDs[1]).WillReturnRows(fakeResult)

			res, err := reader.GetUserVideoPrivacyPropertiesBatch(context.Background(), userIDs)

			expected123 := edgemodels.UserVideoPrivacyProperties{
				UserID:       123,
				HideArchives: false,
				UpdatedAt: now,
				CreatedAt: now,
			}
			expected124 := edgemodels.UserVideoPrivacyProperties{
				UserID:       124,
				HideArchives: true,
				UpdatedAt: now,
				CreatedAt: now,
			}
			So(res[123], ShouldResemble, expected123)
			So(res[124], ShouldResemble, expected124)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

		Convey("Returns default props if it doesn't exist", func() {
			userIDs := []int64{ int64(123), int64(124) }
			res, err := reader.GetUserVideoPrivacyPropertiesBatch(context.Background(), userIDs)
			query := `SELECT user_id,hide_archives,created_at,updated_at FROM user_video_privacy_properties WHERE user_id IN ($1, $2)`
			mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(userIDs[0], userIDs[1]).WillReturnError(sql.ErrNoRows)
			expected123 := edgemodels.UserVideoPrivacyProperties{UserID: 123}
			expected124 := edgemodels.UserVideoPrivacyProperties{UserID: 124}
			So(res[123], ShouldResemble, expected123)
			So(res[124], ShouldResemble, expected124)
			So(err, ShouldBeNil)
		})

	})
}
