package vinyldb

import (
	"database/sql"
	"testing"
	"time"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"

	"github.com/DATA-DOG/go-sqlmock"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func TestGetUserVODProperties(t *testing.T) {
	now := time.Now()
	Convey("Fetching props for a user", t, func() {
		mock, reader := Setup()

		Convey("Fetches a user's props if it exists", func() {
			userID := int64(123)
			query := models.FetchUserVODProperties() + ` WHERE user_id = (.+)`
			fakeResult := sqlmock.NewRows(models.UserVODPropertiesFields).AddRow(123, false, 14, true, false, false, now, now)
			mock.ExpectQuery(query).WithArgs(userID).WillReturnRows(fakeResult)
			res, err := reader.GetUserVODProperties(context.Background(), userID)

			expected := &models.UserVodProperties{
				UserID:                   userID,
				SaveVODsForever:          sql.NullBool{Valid: true, Bool: false},
				VODStorageDays:           sql.NullInt64{Valid: true, Int64: 14},
				CanUploadVOD:             sql.NullBool{Valid: true, Bool: true},
				YoutubeExportingDisabled: sql.NullBool{Valid: true, Bool: false},
				SkipUploadModeration:     sql.NullBool{Valid: true, Bool: false},
				UpdatedAt:                now,
				CreatedAt:                now,
			}
			So(res, ShouldResemble, expected)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

		Convey("Returns default props if it doesn't exist", func() {
			userID := int64(123)
			query := models.FetchUserVODProperties() + ` WHERE user_id = (.+)`
			mock.ExpectQuery(query).WithArgs(userID).WillReturnError(sql.ErrNoRows)
			res, err := reader.GetUserVODProperties(context.Background(), userID)

			expected := &models.UserVodProperties{UserID: userID}
			So(res, ShouldResemble, expected)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

	})
}
