package vinyldb

import (
	"fmt"

	"golang.org/x/net/context"

	"code.justin.tv/chat/db"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	edgeModels "code.justin.tv/vod/vinyl/models"
)

// GetVodAppeals creates a query and sends it to the datastore and returns the
// corresponding vod appeals that match the query
func (b *Backend) GetVodAppeals(ctx context.Context, priority, resolved edgeModels.NullBool, userInfo string, vodID edgeModels.NullInt64, limit, offset int) ([]*models.VodAppeal, error) {
	queryParts := []interface{}{models.FetchVODAppeals()}
	fields, values := createGetVodAppealsParamsQuery(priority, resolved, userInfo, vodID, "deleted is not true")
	queryParts = append(queryParts, fields...)
	queryParts = append(queryParts, fmt.Sprintf("ORDER BY %s.id DESC", models.VodTableName), "LIMIT", db.Param, "OFFSET", db.Param)

	values = append(values, limit, offset)
	query := db.BuildQuery(queryParts...)
	rows, err := b.conn.Query(ctx, "get_vod_appeals", query, values...)
	vas, err := models.ReadVodAppealRows(rows, err, b.logger)
	if err != nil {
		return nil, err
	}

	return vas, nil
}

// createGetVodAppealsParamsQuery takes the params passed in and creates a query.
func createGetVodAppealsParamsQuery(priority, resolved edgeModels.NullBool, userInfo string, vodID edgeModels.NullInt64, clauses ...string) ([]interface{}, []interface{}) {
	retFields := []interface{}{"WHERE"}
	retValues := []interface{}{}

	if priority.Valid {
		retFields = append(retFields, "priority =", db.Param, "AND")
		retValues = append(retValues, priority.Bool)
	}

	if resolved.Valid {
		if resolved.Bool {
			retFields = append(retFields, "resolved_at IS NOT NULL", "AND")
		} else {
			retFields = append(retFields, "resolved_at IS NULL", "AND")
		}
	}

	if userInfo != "" {
		// user_info is a match as long as one of the fields contains the string
		retFields = append(retFields,
			fmt.Sprintf("(full_name LIKE '%%%s%%' OR", userInfo),
			fmt.Sprintf("street_address_1 LIKE '%%%s%%' OR", userInfo),
			fmt.Sprintf("street_address_2 LIKE '%%%s%%' OR", userInfo),
			fmt.Sprintf("city LIKE '%%%s%%' OR", userInfo),
			fmt.Sprintf("state LIKE '%%%s%%' OR", userInfo),
			fmt.Sprintf("zipcode LIKE '%%%s%%' OR", userInfo),
			fmt.Sprintf("country LIKE '%%%s%%')", userInfo),
			"AND",
		)
	}

	if vodID.Valid {
		retFields = append(retFields, "vod_id =", db.Param, "AND")
		retValues = append(retValues, vodID.Int64)
	}

	for _, clause := range clauses {
		retFields = append(retFields, clause, "AND")
	}

	// remove the last AND or WHERE if no fields are added
	retFields = retFields[0 : len(retFields)-1]
	return retFields, retValues
}
