package vinyldb

import (
	"regexp"
	"testing"

	"golang.org/x/net/context"

	. "github.com/smartystreets/goconvey/convey"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	edgeModels "code.justin.tv/vod/vinyl/models"
)

func TestGetVodAppeals(t *testing.T) {
	Convey("GetVodAppeals", t, func() {
		mock, reader := Setup()
		priority := edgeModels.NullBool{}
		resolved := edgeModels.NullBool{}
		userInfo := ""
		vodID := edgeModels.NullInt64{}
		limit := 10
		offset := 0

		Convey("gets the appropriate vod appeals", func() {
			Convey("with no optional params", func() {
				query := models.FetchVODAppeals() +
					" WHERE deleted is not true" +
					` ORDER BY vods.id DESC LIMIT $1 OFFSET $2`
				mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(limit, offset)

				_, err := reader.GetVodAppeals(context.Background(), priority, resolved, userInfo, vodID, limit, offset)

				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

			Convey("with priority", func() {
				priority := edgeModels.NullBool{Bool: true, Valid: true}

				query := models.FetchVODAppeals() +
					` WHERE priority = $1 AND deleted is not true ORDER BY vods.id DESC LIMIT $2 OFFSET $3`
				mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(priority.Bool, limit, offset)

				_, err := reader.GetVodAppeals(context.Background(), priority, resolved, userInfo, vodID, limit, offset)

				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

			Convey("with resolved", func() {
				resolved := edgeModels.NullBool{Bool: true, Valid: true}

				query := models.FetchVODAppeals() +
					` WHERE resolved_at IS NOT NULL AND deleted is not true ORDER BY vods.id DESC LIMIT $1 OFFSET $2`
				mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(limit, offset)

				_, err := reader.GetVodAppeals(context.Background(), priority, resolved, userInfo, vodID, limit, offset)

				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

			Convey("with user_info", func() {
				userInfo := "Alice"

				query := models.FetchVODAppeals() +
					` WHERE (full_name LIKE '%` + userInfo + `%' OR` +
					` street_address_1 LIKE '%` + userInfo + `%' OR` +
					` street_address_2 LIKE '%` + userInfo + `%' OR` +
					` city LIKE '%` + userInfo + `%' OR` +
					` state LIKE '%` + userInfo + `%' OR` +
					` zipcode LIKE '%` + userInfo + `%' OR` +
					` country LIKE '%` + userInfo + `%')` +
					` AND deleted is not true` +
					` ORDER BY vods.id DESC` +
					` LIMIT $1 OFFSET $2`
				mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(limit, offset)

				_, err := reader.GetVodAppeals(context.Background(), priority, resolved, userInfo, vodID, limit, offset)

				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

			Convey("with vod_id", func() {
				vodID := edgeModels.NullInt64{Int64: int64(1234), Valid: true}

				query := models.FetchVODAppeals() +
					` WHERE vod_id = $1 AND deleted is not true ORDER BY vods.id DESC LIMIT $2 OFFSET $3`
				mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(vodID.Int64, limit, offset)

				_, err := reader.GetVodAppeals(context.Background(), priority, resolved, userInfo, vodID, limit, offset)

				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

			Convey("with all params", func() {
				priority := edgeModels.NullBool{Bool: true, Valid: true}
				resolved := edgeModels.NullBool{Bool: false, Valid: true}
				userInfo := "Bob"
				vodID := edgeModels.NullInt64{Int64: int64(2345), Valid: true}

				query := models.FetchVODAppeals() +
					` WHERE priority = $1 AND resolved_at IS NULL AND` +
					` (full_name LIKE '%` + userInfo + `%' OR` +
					` street_address_1 LIKE '%` + userInfo + `%' OR` +
					` street_address_2 LIKE '%` + userInfo + `%' OR` +
					` city LIKE '%` + userInfo + `%' OR` +
					` state LIKE '%` + userInfo + `%' OR` +
					` zipcode LIKE '%` + userInfo + `%' OR` +
					` country LIKE '%` + userInfo + `%') AND` +
					` vod_id = $2` +
					` AND deleted is not true` +
					` ORDER BY vods.id DESC LIMIT $3 OFFSET $4`
				mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(priority.Bool, vodID.Int64, limit, offset)

				_, err := reader.GetVodAppeals(context.Background(), priority, resolved, userInfo, vodID, limit, offset)

				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
		})
	})
}
