package vinyldb

import (
	"regexp"
	"testing"
	"time"

	sqlmock "github.com/DATA-DOG/go-sqlmock"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	edge "code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func TestGetVodsByID(t *testing.T) {
	Convey("Fetching vods by ID", t, func() {
		mock, reader := Setup()
		ids := []int64{123, 456}
		Convey("Should query the db with the right parameters.", func() {
			query := models.FetchAllVODFieldsQuery() + ` WHERE id IN (.+) AND deleted IS NOT TRUE`

			now := time.Now()
			fakeResult := sqlmock.NewRows(models.VodFields)
			fakeResult.AddRow(
				1, nil, 34, "archive", nil, nil,
				now, nil, nil, nil, nil, nil, nil,
				1238, nil, nil, nil, nil, nil,
				0, 111, nil, nil, nil, nil, nil,
				now, edge.StatusRecording, nil, nil, now,
				"hello", 0, nil, "", nil)

			mock.ExpectQuery(query).WithArgs(123, 456).WillReturnRows(fakeResult)

			fakeVodThumbnails := sqlmock.NewRows([]string{"id", "vod_id", "path", "offset", "type"})
			mock.ExpectQuery(regexp.QuoteMeta(`SELECT id, vod_id, path, "offset", type FROM vod_thumbnails WHERE vod_id IN ( 1 )`)).WillReturnRows(fakeVodThumbnails)

			_, err := reader.GetVodsByID(context.Background(), ids, edge.VodFilterDeleted())
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

		Convey("finds a deleted vod when activeOnly and viewableOnly are false", func() {
			query := models.FetchAllVODFieldsQuery() + ` WHERE id IN (.+)`

			now := time.Now()
			fakeResult := sqlmock.NewRows(models.VodFields)
			fakeResult.AddRow(
				1, nil, 34, "archive", nil, nil,
				now, nil, true, nil, nil, nil, nil,
				1238, nil, nil, nil, nil, nil,
				0, 111, nil, nil, nil, nil, nil,
				now, edge.StatusRecording, nil, nil, now,
				"hello", 0, nil, "", nil)

			mock.ExpectQuery(query).WithArgs(123, 456).WillReturnRows(fakeResult)

			fakeVodThumbnails := sqlmock.NewRows([]string{"id", "vod_id", "path", "offset", "type"})
			mock.ExpectQuery(regexp.QuoteMeta(`SELECT id, vod_id, path, "offset", type FROM vod_thumbnails WHERE vod_id IN ( 1 )`)).WillReturnRows(fakeVodThumbnails)

			_, err := reader.GetVodsByID(context.Background(), ids, edge.VodFilterNone())
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

	})
}
