package vinyldb

import (
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/chat/db"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
)

// createParamsQuery takes the params passed in and creates a query. broadcastType,
// startTime, and endTime are optional params. startTime and endTime are empty if
// it passes time.Time.IsZero().
func createParamsQuery(status string, broadcastType []string, startTime, endTime time.Time) ([]interface{}, []interface{}) {
	// pass in status first as it is a require param
	retFields := []interface{}{"status", "=", db.Param, "AND"}
	retValues := []interface{}{status}

	for _, b := range broadcastType {
		retValues = append(retValues, b)
	}
	retFields = append(retFields, "broadcast_type IN ", db.Params(len(broadcastType)), "AND")

	if !startTime.IsZero() {
		retFields = append(retFields, "(started_on", ">", db.Param, ")", "AND")
		retValues = append(retValues, startTime)
	}

	if !endTime.IsZero() {
		retFields = append(retFields, "(started_on", "<", db.Param, ")", "AND")
		retValues = append(retValues, endTime)
	}

	// remove the last AND
	retFields = retFields[0 : len(retFields)-1]
	return retFields, retValues
}

// GetVodsByStatus creates a query and sends it to the datastore and returns the
// corresponding vods that match the query
func (b *Backend) GetVodsByStatus(ctx context.Context, status string, broadcastType []string, startTime, endTime time.Time) ([]*models.Vod, error) {
	queryParts := []interface{}{models.FetchAllVODFieldsQuery(), "WHERE"}
	fields, values := createParamsQuery(status, broadcastType, startTime, endTime)
	queryParts = append(queryParts, fields...)
	query := db.BuildQuery(queryParts...)

	rows, err := b.conn.Query(ctx, "get_vods_by_status", query, values...)
	if err != nil {
		return nil, err
	}
	res, err := models.ReadVodRows(rows, err, b.logger)
	if err != nil {
		return nil, err
	}
	err = b.AttachVodThumbnails(ctx, res)
	if err != nil {
		return nil, err
	}
	return res, nil
}
