package vinyldb

import (
	"regexp"
	"testing"
	"time"

	. "github.com/smartystreets/goconvey/convey"

	"golang.org/x/net/context"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
)

func TestGetVodsByStatus(t *testing.T) {
	Convey("GetVodsByStatus", t, func() {
		mock, reader := Setup()

		Convey("gets the appropriate vods by status in the table", func() {
			var status string
			var broadcastType []string
			var startTime, endTime time.Time
			Convey("with no start or end times", func() {
				status = "unprocessed"
				broadcastType = []string{"highlight"}
				startTime = time.Time{}
				endTime = time.Time{}

				query := regexp.QuoteMeta(models.FetchAllVODFieldsQuery() + ` WHERE status = $1 AND broadcast_type IN ($2)`)
				mock.ExpectQuery(query).WithArgs(status, broadcastType[0])

				_, err := reader.GetVodsByStatus(context.Background(), status, broadcastType, startTime, endTime)
				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

			Convey("with a start time", func() {
				status = "unprocessed"
				broadcastType = []string{"highlight"}
				startTime = time.Now().UTC()
				endTime = time.Time{}

				query := models.FetchAllVODFieldsQuery() +
					` WHERE status = $1 AND` +
					` broadcast_type IN ($2) AND` +
					` (started_on > $3 )`
				mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(status, broadcastType[0], startTime)

				_, err := reader.GetVodsByStatus(context.Background(), status, broadcastType, startTime, endTime)
				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

			Convey("with an end time", func() {
				status = "unprocessed"
				broadcastType = []string{"highlight"}
				startTime = time.Time{}
				endTime = time.Now().UTC()

				query := models.FetchAllVODFieldsQuery() +
					` WHERE status = $1 AND` +
					` broadcast_type IN ($2) AND` +
					` (started_on < $3 )`
				mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(status, broadcastType[0], endTime)

				_, err := reader.GetVodsByStatus(context.Background(), status, broadcastType, startTime, endTime)
				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

			Convey("with multiple broadcast types", func() {
				status = "unprocessed"
				broadcastType = []string{"highlight", "upload"}
				startTime = time.Time{}
				endTime = time.Time{}

				query := models.FetchAllVODFieldsQuery() +
					` WHERE status = $1 AND` +
					` broadcast_type IN ($2, $3)`
				mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(status, broadcastType[0], broadcastType[1])

				_, err := reader.GetVodsByStatus(context.Background(), status, broadcastType, startTime, endTime)
				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

			Convey("with all params", func() {
				status = "unprocessed"
				broadcastType = []string{"highlight"}
				startTime = time.Now()
				endTime = time.Now().UTC()

				query := models.FetchAllVODFieldsQuery() +
					` WHERE status = $1 AND` +
					` broadcast_type IN ($2) AND` +
					` (started_on > $3 ) AND` +
					` (started_on < $4 )`
				mock.ExpectQuery(regexp.QuoteMeta(query)).WithArgs(status, broadcastType[0], startTime, endTime)

				_, err := reader.GetVodsByStatus(context.Background(), status, broadcastType, startTime, endTime)
				So(mock.ExpectationsWereMet(), ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
		})
	})
}
