package vinyldb

import (
	"fmt"

	"code.justin.tv/chat/db"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	edge "code.justin.tv/vod/vinyl/models"
	"golang.org/x/net/context"
)

var byUserPGSortValues = map[string]string{
	"time":  "COALESCE(published_at,created_at) DESC, created_at DESC",
	"views": "views_count DESC",
}

func byUserPGSortQueryPart(sort string) string {
	return fmt.Sprintf("ORDER BY %s", byUserPGSortValues[sort])
}

// GetVodsByUser calls vinylDB to find the vods for a user with the relevant broadcast type.
func (b *Backend) GetVodsByUser(ctx context.Context, userID int, broadcastType, language, status []string, filters *edge.VodFilters, sort string, limit, offset int) ([]*models.Vod, error) {
	res := []*models.Vod{}

	queryParts := []interface{}{
		models.FetchAllVODFieldsQuery(),
		"WHERE owner_id =", db.Param,
	}
	values := []interface{}{userID}

	for _, b := range broadcastType {
		values = append(values, b)
	}
	queryParts = append(queryParts, "AND broadcast_type IN ", db.Params(len(broadcastType)))

	if len(language) > 0 {
		for _, l := range language {
			values = append(values, l)
		}
		queryParts = append(queryParts, "AND language IN ", db.Params(len(language)))
	}

	if len(status) > 0 {
		for _, s := range status {
			values = append(values, s)
		}
		queryParts = append(queryParts, "AND status IN ", db.Params(len(status)))
	}

	queryParts = models.AppendVodFilters(queryParts, filters)

	queryParts = append(queryParts, byUserPGSortQueryPart(sort))
	queryParts = append(queryParts, "LIMIT", db.Param, "OFFSET", db.Param)
	values = append(values, limit, offset)

	query := db.BuildQuery(queryParts...)

	rows, err := b.conn.Query(ctx, "get_vods_for_user", query, values...)
	res, err = models.ReadVodRows(rows, err, b.logger)
	if err != nil {
		return nil, err
	}
	err = b.AttachVodThumbnails(ctx, res)
	if err != nil {
		return nil, err
	}

	return res, nil
}
