package vinyldb

import (
	"testing"

	sqlmock "github.com/DATA-DOG/go-sqlmock"

	"code.justin.tv/vod/vinyl/datastore/vinyldb/models"
	edge "code.justin.tv/vod/vinyl/models"

	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func TestGetVodsByUser(t *testing.T) {
	Convey("Fetching vods by user", t, func() {
		mock, reader := Setup()
		defaultLimit := 10
		defaultOffset := 0
		userID := 123
		language := []string{}
		sort := "time"

		Convey("should query the db for 1 broadcast type", func() {
			fakeResult := sqlmock.NewRows(models.VodFields)
			broadcastType := []string{"highlight"}
			status := []string{edge.StatusTranscoding}
			query := `SELECT (.+) FROM vods WHERE owner_id = (.+) AND broadcast_type IN (.+) AND status IN (.+) AND deleted IS NOT TRUE ORDER BY COALESCE\(published_at,created_at\) DESC, created_at DESC LIMIT (.+) OFFSET (.+)`
			mock.ExpectQuery(query).WithArgs(userID, broadcastType[0], status[0], defaultLimit, defaultOffset).WillReturnRows(fakeResult)

			_, err := reader.GetVodsByUser(context.Background(), userID, broadcastType, language, status, edge.VodFilterDeleted(), sort, defaultLimit, defaultOffset)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

		Convey("should query the db for the relevant types when broadcast_type is passed in", func() {
			fakeResult := sqlmock.NewRows(models.VodFields)
			broadcastType := []string{"archive", "upload"}
			status := []string{}
			query := `SELECT (.+) FROM vods WHERE owner_id = (.+) AND broadcast_type IN (.+) AND deleted IS NOT TRUE ORDER BY COALESCE\(published_at,created_at\) DESC, created_at DESC LIMIT (.+) OFFSET (.+)`
			mock.ExpectQuery(query).WithArgs(userID, broadcastType[0], broadcastType[1], defaultLimit, defaultOffset).WillReturnRows(fakeResult)

			_, err := reader.GetVodsByUser(context.Background(), userID, broadcastType, language, status, edge.VodFilterDeleted(), sort, defaultLimit, defaultOffset)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

		Convey("should include deleted vods when includeDeleted is true", func() {
			fakeResult := sqlmock.NewRows(models.VodFields)
			broadcastType := []string{"archive"}
			status := []string{"hi", "hey"}
			query := `SELECT (.+) FROM vods WHERE owner_id = (.+) AND broadcast_type IN (.+) AND status IN (.+) ORDER BY COALESCE\(published_at,created_at\) DESC, created_at DESC LIMIT (.+) OFFSET (.+)`
			mock.ExpectQuery(query).WithArgs(userID, broadcastType[0], status[0], status[1], defaultLimit, defaultOffset).WillReturnRows(fakeResult)

			_, err := reader.GetVodsByUser(context.Background(), userID, broadcastType, language, status, edge.VodFilterNone(), sort, defaultLimit, defaultOffset)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

		Convey("should include language", func() {
			fakeResult := sqlmock.NewRows(models.VodFields)
			broadcastType := []string{"highlight"}
			language := []string{"en"}
			status := []string{edge.StatusTranscoding}
			query := `SELECT (.+) FROM vods WHERE owner_id = (.+) AND broadcast_type IN (.+) AND language IN (.+) AND status IN (.+) AND deleted IS NOT TRUE ORDER BY COALESCE\(published_at,created_at\) DESC, created_at DESC LIMIT (.+) OFFSET (.+)`
			mock.ExpectQuery(query).WithArgs(userID, broadcastType[0], language[0], status[0], defaultLimit, defaultOffset).WillReturnRows(fakeResult)

			_, err := reader.GetVodsByUser(context.Background(), userID, broadcastType, language, status, edge.VodFilterDeleted(), sort, defaultLimit, defaultOffset)
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
		})

	})
}
