package vinyldb

import (
	"testing"

	"code.justin.tv/vod/vinyl/models"

	sqlmock "github.com/DATA-DOG/go-sqlmock"

	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func TestGetVodsCountByUser(t *testing.T) {
	Convey("Fetching vod counts by user", t, func() {
		mock, reader := Setup()
		userID := 123
		language := []string{}

		Convey("should query the db for all types when the broadcast_type is all", func() {
			fakeResult := sqlmock.NewRows([]string{"count"})
			fakeResult.AddRow(1)
			broadcastType := []string{"highlight"}
			status := []string{models.StatusRecorded}
			query := `SELECT COUNT\(1\) FROM vods WHERE owner_id = (.+) AND broadcast_type IN (.+) AND status IN (.+) AND deleted IS NOT TRUE`
			mock.ExpectQuery(query).WithArgs(userID, broadcastType[0], status[0]).WillReturnRows(fakeResult)

			count, err := reader.GetVodsCountByUser(context.Background(), userID, broadcastType, language, status, models.VodFilterDeleted())
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
			So(count, ShouldEqual, 1)
		})

		Convey("multiple broadcast_types", func() {
			fakeResult := sqlmock.NewRows([]string{"count"})
			fakeResult.AddRow(1)
			broadcastType := []string{"archive", "upload"}
			status := []string{models.StatusRecording, models.StatusRecorded}
			query := `SELECT COUNT\(1\) FROM vods WHERE owner_id = (.+) AND broadcast_type IN (.+) AND status IN (.+) AND deleted IS NOT TRUE`
			mock.ExpectQuery(query).WithArgs(userID, broadcastType[0], broadcastType[1], status[0], status[1]).WillReturnRows(fakeResult)

			count, err := reader.GetVodsCountByUser(context.Background(), userID, broadcastType, language, status, models.VodFilterDeleted())
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
			So(count, ShouldEqual, 1)
		})

		Convey("should include deleted vods when includeDeleted is true", func() {
			fakeResult := sqlmock.NewRows([]string{"count"})
			fakeResult.AddRow(1)
			broadcastType := []string{"archive"}
			status := []string{models.StatusCreated}
			query := `SELECT COUNT\(1\) FROM vods WHERE owner_id = (.+) AND broadcast_type IN (.+) AND status IN (.+)`
			mock.ExpectQuery(query).WithArgs(userID, broadcastType[0], status[0]).WillReturnRows(fakeResult)

			count, err := reader.GetVodsCountByUser(context.Background(), userID, broadcastType, language, status, models.VodFilterNone())
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
			So(count, ShouldEqual, 1)
		})

		Convey("with language", func() {
			fakeResult := sqlmock.NewRows([]string{"count"})
			fakeResult.AddRow(1)
			broadcastType := []string{"archive"}
			language = []string{"en"}
			status := []string{models.StatusCreated}
			query := `SELECT COUNT\(1\) FROM vods WHERE owner_id = (.+) AND broadcast_type IN (.+) AND language IN (.+) AND status IN (.+)`
			mock.ExpectQuery(query).WithArgs(userID, broadcastType[0], language[0], status[0]).WillReturnRows(fakeResult)

			count, err := reader.GetVodsCountByUser(context.Background(), userID, broadcastType, language, status, models.VodFilterNone())
			So(mock.ExpectationsWereMet(), ShouldBeNil)
			So(err, ShouldBeNil)
			So(count, ShouldEqual, 1)
		})

	})
}
