package models

import (
	"database/sql"
	"strings"
	"time"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/utils"
	"code.justin.tv/vod/vinyl/models"
	"github.com/lib/pq"
)

// Constants used to keep track of the audible magic responses table and its fields.
var (
	AMRUpdateFields = []string{
		"title", "performer", "genre", "artist", "album_title",
		"song", "isrc", "is_match", "match_offset_seconds", "match_duration_seconds",
		"scan_offset_seconds", "scan_duration_seconds", "mute_offset_seconds",
		"mute_duration_seconds", "created_at", "updated_at", "song_length_seconds",
		"unmuted_at", "am_item_id",
	}
	AMRCreateFields = append([]string{"vod_id"}, AMRUpdateFields...)
	AMRFields       = append([]string{"id"}, AMRCreateFields...)
	AMRTableName    = "audible_magic_responses"
)

// FetchAMRQuery returns a string used to grab all fields from the audible magic responses table.
func FetchAMRQuery() string {
	return "SELECT " + strings.Join(AMRFields, ",") + " FROM " + AMRTableName
}

// FetchAMRSegmentsForVod returns a string used to fetch audible magic responss segments for a vod.
func FetchAMRSegmentsForVod(vodID int64) string {
	return db.BuildQuery(
		"SELECT mute_offset_seconds, mute_duration_seconds",
		"FROM "+AMRTableName,
		"WHERE vod_id = $1")
}

// AMR is the representation for a VinylDB audible magic response.
type AMR struct {
	AudibleMagicResponseID int64
	VodID                  int64
	Title                  sql.NullString
	Performer              sql.NullString
	Genre                  sql.NullString
	Artist                 sql.NullString
	AlbumTitle             sql.NullString
	Song                   sql.NullString
	Isrc                   sql.NullString
	IsMatch                bool
	MatchOffsetSeconds     sql.NullInt64
	MatchDurationSeconds   sql.NullInt64
	ScanOffsetSeconds      sql.NullInt64
	ScanDurationSeconds    sql.NullInt64
	MuteOffsetSeconds      sql.NullInt64
	MuteDurationSeconds    sql.NullInt64
	CreatedAt              time.Time
	UpdatedAt              time.Time
	SongLength             sql.NullInt64
	UnmutedAt              pq.NullTime
	AmItemID               sql.NullString
}

// MutedSegment is the representation for a VinylDB muted segment.
type MutedSegment struct {
	VodID               int64
	MuteOffsetSeconds   sql.NullInt64
	MuteDurationSeconds sql.NullInt64
}

// FromVinylAMR converts a VinylAMR into a AMR to be used in the vinyldb package.
func FromVinylAMR(vinylAMR *models.AMR) (*AMR, error) {
	return &AMR{
		AudibleMagicResponseID: vinylAMR.AudibleMagicResponseID,
		VodID:                vinylAMR.VodID,
		Title:                sql.NullString(vinylAMR.Title),
		Performer:            sql.NullString(vinylAMR.Performer),
		Genre:                sql.NullString(vinylAMR.Genre),
		Artist:               sql.NullString(vinylAMR.Artist),
		AlbumTitle:           sql.NullString(vinylAMR.AlbumTitle),
		Song:                 sql.NullString(vinylAMR.Song),
		Isrc:                 sql.NullString(vinylAMR.Isrc),
		IsMatch:              vinylAMR.IsMatch,
		MatchOffsetSeconds:   sql.NullInt64(vinylAMR.MatchOffsetSeconds),
		MatchDurationSeconds: sql.NullInt64(vinylAMR.MatchDurationSeconds),
		ScanOffsetSeconds:    sql.NullInt64(vinylAMR.ScanOffsetSeconds),
		ScanDurationSeconds:  sql.NullInt64(vinylAMR.ScanDurationSeconds),
		MuteOffsetSeconds:    sql.NullInt64(vinylAMR.MuteOffsetSeconds),
		MuteDurationSeconds:  sql.NullInt64(vinylAMR.MuteDurationSeconds),
		CreatedAt:            vinylAMR.CreatedAt.Round(time.Second),
		UpdatedAt:            vinylAMR.UpdatedAt.Round(time.Second),
		SongLength:           sql.NullInt64(vinylAMR.SongLength),
		UnmutedAt:            vinylAMR.UnmutedAt.AsPQ(),
		AmItemID:             sql.NullString(vinylAMR.AmItemID),
	}, nil
}

// AsVinylAMR converts a AMR object into a VinylAMR object to be returned.
func (A *AMR) AsVinylAMR() (*models.AMR, error) {
	return &models.AMR{
		AudibleMagicResponseID: A.AudibleMagicResponseID,
		VodID:                A.VodID,
		Title:                models.NullString(A.Title),
		Performer:            models.NullString(A.Performer),
		Genre:                models.NullString(A.Genre),
		Artist:               models.NullString(A.Artist),
		AlbumTitle:           models.NullString(A.AlbumTitle),
		Song:                 models.NullString(A.Song),
		Isrc:                 models.NullString(A.Isrc),
		IsMatch:              A.IsMatch,
		MatchOffsetSeconds:   models.NullInt64(A.MatchOffsetSeconds),
		MatchDurationSeconds: models.NullInt64(A.MatchDurationSeconds),
		ScanOffsetSeconds:    models.NullInt64(A.ScanOffsetSeconds),
		ScanDurationSeconds:  models.NullInt64(A.ScanDurationSeconds),
		MuteOffsetSeconds:    models.NullInt64(A.MuteOffsetSeconds),
		MuteDurationSeconds:  models.NullInt64(A.MuteDurationSeconds),
		CreatedAt:            A.CreatedAt,
		UpdatedAt:            A.UpdatedAt,
		SongLength:           models.NullInt64(A.SongLength),
		UnmutedAt:            models.PQAsNullTime(A.UnmutedAt),
		AmItemID:             models.NullString(A.AmItemID)}, nil
}

// ValuesList returns a list of the field values associated with a AMR.
func (A *AMR) ValuesList() []interface{} {
	return []interface{}{
		A.VodID,
		A.Title,
		A.Performer,
		A.Genre,
		A.Artist,
		A.AlbumTitle,
		A.Song,
		A.Isrc,
		A.IsMatch,
		A.MatchOffsetSeconds,
		A.MatchDurationSeconds,
		A.ScanOffsetSeconds,
		A.ScanDurationSeconds,
		A.MuteOffsetSeconds,
		A.MuteDurationSeconds,
		A.CreatedAt,
		A.UpdatedAt,
		A.SongLength,
		A.UnmutedAt,
		A.AmItemID,
	}
}

// ReadAMRRows converts the results of a query into a list of AMR objects to be converted into VinylAMRs.
func ReadAMRRows(rows db.Rows, queryErr error, logger errorlogger.ErrorLogger) ([]*AMR, error) {
	res := []*AMR{}

	if queryErr == sql.ErrNoRows {
		return res, nil
	}

	if queryErr != nil {
		return nil, queryErr
	}

	defer utils.CloseRows(rows, logger)

	for rows.Next() {
		amr := AMR{}
		if err := rows.Scan(
			&amr.AudibleMagicResponseID,
			&amr.VodID,
			&amr.Title,
			&amr.Performer,
			&amr.Genre,
			&amr.Artist,
			&amr.AlbumTitle,
			&amr.Song,
			&amr.Isrc,
			&amr.IsMatch,
			&amr.MatchOffsetSeconds,
			&amr.MatchDurationSeconds,
			&amr.ScanOffsetSeconds,
			&amr.ScanDurationSeconds,
			&amr.MuteOffsetSeconds,
			&amr.MuteDurationSeconds,
			&amr.CreatedAt,
			&amr.UpdatedAt,
			&amr.SongLength,
			&amr.UnmutedAt,
			&amr.AmItemID); err != nil {
			return nil, err
		}

		res = append(res, &amr)
	}
	return res, nil
}
