package models

import (
	"database/sql"
	"strings"
	"time"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/utils"
	"code.justin.tv/vod/vinyl/models"
	"github.com/lib/pq"
)

// Constants used to keep track of the track appeals table and it's fields.
var (
	TrackAppealCreateFields = []string{
		"audible_magic_response_id", "vod_appeal_id",
		"reason", "created_at", "updated_at", "resolved_at",
	}

	TrackAppealFields     = append([]string{"id"}, TrackAppealCreateFields...)
	TrackAppealsTableName = "track_appeals"
)

// TrackAppeal represents the VinylDB structure of track appeals.
type TrackAppeal struct {
	TrackAppealID          int64
	AudibleMagicResponseID int64
	VodAppealID            int64
	Reason                 string
	CreatedAt              time.Time
	UpdatedAt              time.Time
	ResolvedAt             pq.NullTime
}

// FetchTrackAppeals returns a string used to fetch all columns of a track appeal.
func FetchTrackAppeals() string {
	return "SELECT " + strings.Join(TrackAppealFields, ",") + " FROM " + TrackAppealsTableName
}

// FromVinylTrackAppeal converts a Vinyl track appeal to a VinylDB track appeal object.
func FromVinylTrackAppeal(vinylTrackAppeal *models.TrackAppeal) (*TrackAppeal, error) {
	return &TrackAppeal{
		TrackAppealID:          vinylTrackAppeal.TrackAppealID,
		AudibleMagicResponseID: vinylTrackAppeal.AudibleMagicResponseID,
		VodAppealID:            vinylTrackAppeal.VodAppealID,
		Reason:                 vinylTrackAppeal.Reason,
		CreatedAt:              vinylTrackAppeal.CreatedAt.Round(time.Second),
		UpdatedAt:              vinylTrackAppeal.CreatedAt.Round(time.Second),
		ResolvedAt:             vinylTrackAppeal.ResolvedAt.AsPQ(),
	}, nil
}

// AsVinylTrackAppeal converts a VinylDB track appeal to a Vinyl track appeal to be returned.
func (T *TrackAppeal) AsVinylTrackAppeal() (*models.TrackAppeal, error) {
	resolvedAt := pq.NullTime{Valid: T.ResolvedAt.Valid}
	if T.ResolvedAt.Valid {
		resolvedAt.Time = T.ResolvedAt.Time.Round(time.Second)
	}

	return &models.TrackAppeal{
		TrackAppealID:          T.TrackAppealID,
		AudibleMagicResponseID: T.AudibleMagicResponseID,
		VodAppealID:            T.VodAppealID,
		Reason:                 T.Reason,
		CreatedAt:              T.CreatedAt.Round(time.Second),
		UpdatedAt:              T.CreatedAt.Round(time.Second),
		ResolvedAt:             models.PQAsNullTime(resolvedAt),
	}, nil
}

// ValuesList returns a list of the field values associated with a track appeal.
func (T *TrackAppeal) ValuesList() []interface{} {
	return []interface{}{
		T.AudibleMagicResponseID,
		T.VodAppealID,
		T.Reason,
		T.CreatedAt,
		T.UpdatedAt,
		T.ResolvedAt,
	}
}

// ReadTrackAppealRows converts the results of a query into a list of track appeal objects to be converted into Vinyl track appeals.
func ReadTrackAppealRows(rows db.Rows, queryErr error, logger errorlogger.ErrorLogger) ([]*TrackAppeal, error) {
	res := []*TrackAppeal{}
	if queryErr == sql.ErrNoRows {
		return res, nil
	}

	if queryErr != nil {
		return nil, queryErr
	}

	defer utils.CloseRows(rows, logger)

	for rows.Next() {
		trackAppeal := TrackAppeal{}
		if err := rows.Scan(
			&trackAppeal.TrackAppealID,
			&trackAppeal.AudibleMagicResponseID,
			&trackAppeal.VodAppealID,
			&trackAppeal.Reason,
			&trackAppeal.CreatedAt,
			&trackAppeal.UpdatedAt,
			&trackAppeal.ResolvedAt,
		); err != nil {
			return res, nil
		}

		res = append(res, &trackAppeal)
	}

	return res, nil
}
