package models

import (
	"strings"

	"code.justin.tv/chat/db"
	"code.justin.tv/common/golibs/errorlogger"
	"code.justin.tv/vod/vinyl/datastore/vinyldb/utils"
	"code.justin.tv/vod/vinyl/models"
)

// UserVideoPrivacyPropertiesTableName is the table name for these properties
const UserVideoPrivacyPropertiesTableName = "user_video_privacy_properties"

// Constants associated with the props table and its fields.
var (
	userVideoPrivacyPropertiesEditableFields = []string{"user_id", "hide_archives"}

	UserVideoPrivacyPropertiesFields       = append(userVideoPrivacyPropertiesEditableFields, []string{"created_at", "updated_at"}...)
	UserVideoPrivacyPropertiesUpdateFields = append(userVideoPrivacyPropertiesEditableFields, []string{"updated_at"}...)
)

// FetchUserVideoPrivacyProperties returns a string used to query for all columns in the props table.
func FetchUserVideoPrivacyProperties() string {
	return "SELECT " + strings.Join(UserVideoPrivacyPropertiesFields, ",") + " FROM " + UserVideoPrivacyPropertiesTableName
}

// ReadUserVideoPrivacyPropertiesRow reads the results of a query and turns it into a list of props objects to be returned.
func ReadUserVideoPrivacyPropertiesRow(row db.Row, userID int64) (models.UserVideoPrivacyProperties, error) {
	res := models.UserVideoPrivacyProperties{UserID: userID}

	err := row.Scan(
		&res.UserID,
		&res.HideArchives,
		&res.CreatedAt,
		&res.UpdatedAt,
	)
	if err == db.ErrNoRows {
		return res, nil
	}
	if err != nil {
		return res, err
	}

	return res, nil
}

// ReadUserVideoPrivacyPropertiesRows reads the results of a query and turns it into a list of props objects to be returned.
func ReadUserVideoPrivacyPropertiesRows(rows db.Rows, logger errorlogger.ErrorLogger) ([]models.UserVideoPrivacyProperties, error) {
	defer utils.CloseRows(rows, logger)
	res := []models.UserVideoPrivacyProperties{}

	for rows.Next() {
		props := models.UserVideoPrivacyProperties{}
		if err := rows.Scan(
			&props.UserID,
			&props.HideArchives,
			&props.CreatedAt,
			&props.UpdatedAt,
		); err != nil {
			return nil, err
		}

		res = append(res, props)
	}

	return res, nil
}

// CreateUserVideoPrivacyPropertiesQuery generates the strings for the fields and values for an update sql query
func CreateUserVideoPrivacyPropertiesQuery(userID int64, u models.UserVideoPrivacyPropertiesInput) []interface{} {
	retValues := []interface{}{userID}

	retValues = append(retValues, u.HideArchives.Valid && u.HideArchives.Bool)
	retValues = append(retValues, u.CreatedAt, u.UpdatedAt)

	return retValues
}

// UpdateUserVideoPrivacyPropertiesQuery generates the strings for the fields and values for an update sql query
func UpdateUserVideoPrivacyPropertiesQuery(u models.UserVideoPrivacyPropertiesInput) ([]interface{}, []interface{}) {
	retFields := []interface{}{}
	retValues := []interface{}{}

	if u.HideArchives.Valid {
		retFields = append(retFields, "hide_archives =", db.Param, ",")
		retValues = append(retValues, u.HideArchives.Bool)
	}

	retFields = append(retFields, "updated_at =", db.Param)
	retValues = append(retValues, u.UpdatedAt)

	return retFields, retValues
}
