package models

import (
	"database/sql"
	"fmt"
	"strings"
	"time"

	"code.justin.tv/chat/db"
	edgemodels "code.justin.tv/vod/vinyl/models"
)

// UserVODPropertiesTableName is the table name for these properties
const UserVODPropertiesTableName = "user_vod_properties"

const (
	defaultVODStorageDays = 14
)

var (
	userVODPropertiesEditableFields = []string{
		"user_id",
		"save_vods_forever",
		"vod_storage_days",
		"can_upload_vod",
		"youtube_exporting_disabled",
		"skip_upload_moderation",
	}
	// UserVODPropertiesFields is the list of fields passed to the database for a Create operation
	UserVODPropertiesFields       = append(userVODPropertiesEditableFields, "created_at", "updated_at")
	userVODPropertiesUpdateFields = append(userVODPropertiesEditableFields, "updated_at")
)

// FetchUserVODProperties returns a string used to query for all columns in the table.
func FetchUserVODProperties() string {
	return "SELECT " + strings.Join(UserVODPropertiesFields, ",") + " FROM " + UserVODPropertiesTableName
}

// ReadUserVODPropertiesRow reads the results of a query and turns it into a list of objects to be returned.
func ReadUserVODPropertiesRow(row db.Row, userID int64) (*UserVodProperties, error) {
	res := &UserVodProperties{UserID: userID}

	err := row.Scan(
		&res.UserID,
		&res.SaveVODsForever,
		&res.VODStorageDays,
		&res.CanUploadVOD,
		&res.YoutubeExportingDisabled,
		&res.SkipUploadModeration,
		&res.CreatedAt,
		&res.UpdatedAt,
	)
	if err == db.ErrNoRows {
		return res, nil
	}
	if err != nil {
		fmt.Println(err)
		return res, err
	}
	return res, nil
}

// CreateUserVODPropertiesQuery generates the strings for the fields and values for an update SQL query
func CreateUserVODPropertiesQuery(userID int64, u edgemodels.UserVODPropertiesInput) []interface{} {
	retValues := []interface{}{userID}
	retValues = append(retValues, u.SaveVODsForever.Valid && u.SaveVODsForever.Bool)
	if u.VODStorageDays.Valid {
		retValues = append(retValues, u.VODStorageDays.Int64)
	} else {
		retValues = append(retValues, defaultVODStorageDays)
	}

	retValues = append(retValues, u.CanUploadVOD.Valid && u.CanUploadVOD.Bool)
	retValues = append(retValues, u.YoutubeExportingDisabled.Valid && u.YoutubeExportingDisabled.Bool)
	retValues = append(retValues, u.SkipUploadModeration.Valid && u.SkipUploadModeration.Bool)
	retValues = append(retValues, u.CreatedAt, u.UpdatedAt)

	return retValues
}

// UpdateUserVODPropertiesQuery generates the strings for the fields and values for an update SQL query
func UpdateUserVODPropertiesQuery(u edgemodels.UserVODPropertiesInput) ([]interface{}, []interface{}) {
	retFields := []interface{}{}
	retValues := []interface{}{}

	if u.SaveVODsForever.Valid {
		retFields = append(retFields, "save_vods_forever = ", db.Param, ",")
		retValues = append(retValues, u.SaveVODsForever.Bool)
	}
	if u.VODStorageDays.Valid {
		retFields = append(retFields, "vod_storage_days = ", db.Param, ",")
		retValues = append(retValues, u.VODStorageDays.Int64)
	}
	if u.CanUploadVOD.Valid {
		retFields = append(retFields, "can_upload_vod = ", db.Param, ",")
		retValues = append(retValues, u.CanUploadVOD.Bool)
	}
	if u.YoutubeExportingDisabled.Valid {
		retFields = append(retFields, "youtube_exporting_disabled = ", db.Param, ",")
		retValues = append(retValues, u.YoutubeExportingDisabled.Bool)
	}
	if u.SkipUploadModeration.Valid {
		retFields = append(retFields, "skip_upload_moderation = ", db.Param, ",")
		retValues = append(retValues, u.SkipUploadModeration.Bool)
	}
	retFields = append(retFields, "updated_at = ", db.Param)
	retValues = append(retValues, u.UpdatedAt)

	return retFields, retValues
}

// UserVodProperties is a UserVODProperties implementation
type UserVodProperties struct {
	UserID                   int64
	SaveVODsForever          sql.NullBool
	VODStorageDays           sql.NullInt64
	CanUploadVOD             sql.NullBool
	YoutubeExportingDisabled sql.NullBool
	SkipUploadModeration     sql.NullBool
	CreatedAt                time.Time
	UpdatedAt                time.Time
}

// AsVinylUserVODProperties transforms this UserVodProperties into a VinylUserVODProperties
func (p *UserVodProperties) AsVinylUserVODProperties() (*edgemodels.UserVODProperties, error) {
	res := edgemodels.UserVODProperties{
		UserID:    p.UserID,
		CreatedAt: p.CreatedAt,
		UpdatedAt: p.UpdatedAt,
	}
	// All these fields default to false so no need for additonal logic
	res.SaveVODsForever = p.SaveVODsForever.Valid && p.SaveVODsForever.Bool
	res.CanUploadVOD = true
	res.YoutubeExportingDisabled = p.YoutubeExportingDisabled.Valid && p.YoutubeExportingDisabled.Bool
	res.SkipUploadModeration = p.SkipUploadModeration.Valid && p.SkipUploadModeration.Bool
	if p.VODStorageDays.Valid {
		res.VODStorageDays = int(p.VODStorageDays.Int64)
	} else {
		res.VODStorageDays = defaultVODStorageDays
	}
	return &res, nil
}
